"use strict";

console.debug("background.js");

let lastSavedPayload = null;
let lastSavedTime = 0;

chrome.webRequest.onBeforeRequest.addListener(
    function (details) {
        // console.debug("[LingoPause] Detected URL in webRequest:", details.url);
        const cc_url = details.url;
        const cc_url_obj = new URL(cc_url);
        const pot = cc_url_obj.searchParams.get("pot");
        if (pot) {
            // console.debug("[LingoPause] Captured pot:", pot);
            chrome.storage.session.set({ cc_pot: pot });
        } else {
            console.debug("[LingoPause] No pot found in timedtext URL");
        }
    },
    { urls: ["*://www.youtube.com/api/timedtext*"] },
    []
);

chrome.runtime.onMessage.addListener(function (request, sender, sendResponse) {
    if (request.action === "get-pot") {
        console.debug('background receive "get-pot"');

        (async () => {
            const cc_pot = (await chrome.storage.session.get())['cc_pot'] || "";

            console.debug("send response cc_pot:", cc_pot);

            sendResponse(cc_pot);

        })();

        return true;
    }

    if (request.action === "load-kuroshiro") {
        console.debug('background receive "load-kuroshiro"');

        (async () => {
            try {
                if (!sender.tab || !sender.tab.id) {
                    sendResponse({ success: false, error: "No tab ID" });
                    return;
                }

                const tabId = sender.tab.id;

                console.debug("Loading Kuroshiro scripts from local files...");

                // Check if already loaded
                const checkResult = await chrome.scripting.executeScript({
                    target: { tabId: tabId },
                    world: 'MAIN',
                    func: () => {
                        return {
                            hasKuroshiro: typeof window.Kuroshiro !== 'undefined',
                            hasAnalyzer: typeof window.KuromojiAnalyzer !== 'undefined'
                        };
                    }
                });

                if (checkResult && checkResult[0] && checkResult[0].result &&
                    checkResult[0].result.hasKuroshiro && checkResult[0].result.hasAnalyzer) {
                    console.debug("Kuroshiro already loaded in page context");
                } else {
                    // Inject scripts using files parameter - this bypasses Trusted Types
                    console.debug("Injecting kuroshiro.min.js...");
                    await chrome.scripting.executeScript({
                        target: { tabId: tabId },
                        world: 'MAIN',
                        files: ['kuroshiro.min.js']
                    });

                    console.debug("Injecting kuroshiro-analyzer-kuromoji.min.js...");
                    await chrome.scripting.executeScript({
                        target: { tabId: tabId },
                        world: 'MAIN',
                        files: ['kuroshiro-analyzer-kuromoji.min.js']
                    });

                    // Wait a bit for scripts to initialize
                    await new Promise(resolve => setTimeout(resolve, 1000));

                    // Verify libraries are available with retries
                    let attempts = 0;
                    const maxAttempts = 10;
                    let librariesReady = false;

                    while (attempts < maxAttempts && !librariesReady) {
                        attempts++;
                        const verifyResult = await chrome.scripting.executeScript({
                            target: { tabId: tabId },
                            world: 'MAIN',
                            func: () => {
                                return {
                                    hasKuroshiro: typeof window.Kuroshiro !== 'undefined',
                                    hasAnalyzer: typeof window.KuromojiAnalyzer !== 'undefined',
                                    kuroshiro: typeof window.Kuroshiro,
                                    analyzer: typeof window.KuromojiAnalyzer
                                };
                            }
                        });

                        if (verifyResult && verifyResult[0] && verifyResult[0].result) {
                            const result = verifyResult[0].result;
                            console.debug(`Checking for libraries (attempt ${attempts}/${maxAttempts}):`, result);

                            if (result.hasKuroshiro && result.hasAnalyzer) {
                                console.debug("Both libraries are now available!");
                                librariesReady = true;
                            } else {
                                await new Promise(resolve => setTimeout(resolve, 500));
                            }
                        } else {
                            await new Promise(resolve => setTimeout(resolve, 500));
                        }
                    }

                    if (!librariesReady) {
                        console.error("Libraries not found after maximum attempts");
                    }
                }

                // Set flag that libraries are loaded
                await chrome.scripting.executeScript({
                    target: { tabId: tabId },
                    world: 'MAIN',
                    func: () => {
                        document.documentElement.setAttribute('data-kuroshiro-loaded', 'true');
                    }
                });

                console.debug("Kuroshiro loaded successfully");
                sendResponse({ success: true });
            } catch (error) {
                console.error("Failed to load Kuroshiro:", error);
                sendResponse({ success: false, error: error.message });
            }
        })();

        return true; // Keep channel open for async response
    }

    if (request.action === "convert-japanese") {
        // console.debug('[LingoPause] background receive "convert-japanese", text:', request.text);

        (async () => {
            try {
                if (!sender.tab || !sender.tab.id) {
                    console.error("No tab ID in sender:", sender);
                    sendResponse({ success: false, error: "No tab ID" });
                    return;
                }

                const tabId = sender.tab.id;
                const text = request.text;
                const dictPath = request.dictPath;
                // console.debug("[LingoPause] dictPath:", dictPath);
                // console.debug(`[LingoPause] Executing conversion for tab ${tabId}, text: "${text}"`);

                // Execute conversion in page context where Kuroshiro is available
                const results = await chrome.scripting.executeScript({
                    target: { tabId: tabId },
                    world: 'MAIN',
                    func: (inputText, resolvedDictPath) => {
                        return new Promise(async (resolve) => {
                            try {
                                // console.debug("In page context, checking for Kuroshiro...");
                                // console.debug("resolvedDictPath:", resolvedDictPath);
                                // console.debug("window.Kuroshiro:", typeof window.Kuroshiro);
                                // console.debug("window.Kuroshiro:", window.Kuroshiro);
                                // console.debug("window.KuromojiAnalyzer:", typeof window.KuromojiAnalyzer);
                                // console.debug("window.kuromoji:", typeof window.kuromoji);

                                // Check if libraries are available
                                if (typeof window.Kuroshiro === 'undefined') {
                                    // console.error("window.Kuroshiro is undefined");
                                    resolve(null);
                                    return;
                                }

                                // Handle different export formats (UMD might export default as a property)
                                let KuroshiroClass = window.Kuroshiro;
                                if (window.Kuroshiro && window.Kuroshiro.default) {
                                    KuroshiroClass = window.Kuroshiro.default;
                                    // console.debug("Using window.Kuroshiro.default");
                                } else if (typeof window.Kuroshiro !== 'function') {
                                    // console.error("window.Kuroshiro is not a function:", typeof window.Kuroshiro, window.Kuroshiro);
                                    resolve(null);
                                    return;
                                }

                                // Check for KuromojiAnalyzer - it might be part of Kuroshiro or a separate package
                                let Analyzer = window.KuromojiAnalyzer;
                                if (Analyzer && Analyzer.default) {
                                    Analyzer = Analyzer.default;
                                    // console.debug("Using window.KuromojiAnalyzer.default");
                                }
                                if (!Analyzer && window.Kuroshiro && window.Kuroshiro.Util && window.Kuroshiro.Util.KuromojiAnalyzer) {
                                    Analyzer = window.Kuroshiro.Util.KuromojiAnalyzer;
                                    // console.debug("Found KuromojiAnalyzer in Kuroshiro.Util");
                                } else if (!Analyzer && window.kuromoji) {
                                    // If kuromoji is loaded but KuromojiAnalyzer is not, we might need to use it differently
                                    // console.warn("kuromoji found but KuromojiAnalyzer not found - may need different initialization");
                                }

                                if (!Analyzer || typeof Analyzer !== 'function') {
                                    // console.error("window.KuromojiAnalyzer is undefined or not a function:", typeof Analyzer);
                                    resolve(null);
                                    return;
                                }

                                // console.debug("Kuroshiro libraries found, initializing...");
                                // Initialize if needed (cache in page context)
                                if (!window._kuroshiroInstance) {
                                    // console.debug("Creating new Kuroshiro instance...");
                                    const kuroshiro = new KuroshiroClass();

                                    try {
                                        await kuroshiro.init(new Analyzer({
                                            dictPath: resolvedDictPath
                                        }));
                                        window._kuroshiroInstance = kuroshiro;

                                    } catch (initError) {

                                        throw initError;
                                    }
                                }

                                // console.debug(`Converting text: "${inputText}"`);
                                const result = await window._kuroshiroInstance.convert(inputText, {
                                    to: 'romaji',
                                    mode: 'spaced'
                                });
                                // console.debug(`Conversion result: "${result}"`);
                                resolve(result);
                            } catch (e) {
                                console.error("Kuroshiro conversion error:", e);
                                console.error("Error stack:", e.stack);
                                resolve(null);
                            }
                        });
                    },
                    args: [text, dictPath]
                });

                // console.debug("Script execution results:", results);
                const result = results && results[0] && results[0].result ? results[0].result : null;
                // console.debug(`Final result to send: "${result}"`);
                sendResponse({ success: true, result: result });
            } catch (error) {
                console.error("Failed to convert Japanese text:", error);
                console.error("Error stack:", error.stack);
                sendResponse({ success: false, error: error.message });
            }
        })();

        return true; // Keep channel open for async response
    }

    if (request.action === "SAVE_SUBTITLE") {
        console.debug('[background] Received SAVE_SUBTITLE', request.data);

        // Deduplication/Debounce check
        const now = Date.now();
        if (lastSavedPayload &&
            JSON.stringify(lastSavedPayload) === JSON.stringify(request.data) &&
            (now - lastSavedTime < 5000)) {
            console.debug('[background] Duplicate SAVE_SUBTITLE detected, skipping insert.');
            sendResponse({ success: true, message: "Duplicate suppressed" });
            return true;
        }

        // Lock immediately to prevent race conditions
        lastSavedPayload = request.data;
        lastSavedTime = now;

        (async () => {
            try {
                const { createClient } = self.supabase;
                const { SUPABASE_URL, SUPABASE_ANON_KEY } = self.CONFIG;

                const supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY, {
                    auth: {
                        persistSession: false,
                        autoRefreshToken: true,
                        detectSessionInUrl: false
                    }
                });

                // 2. Get Session from storage
                const { supabase_session } = await chrome.storage.local.get(['supabase_session']);

                if (!supabase_session) {
                    console.warn('[background] No active session found.');
                    sendResponse({ success: false, error: "Not logged in" });
                    return;
                }

                // 3. Set session (with autoRefreshToken: true, this triggers recovery)
                const { error: authError } = await supabase.auth.setSession({
                    access_token: supabase_session.access_token,
                    refresh_token: supabase_session.refresh_token
                });

                if (authError) {
                    console.error('[background] Auth session error:', authError);
                    sendResponse({ success: false, error: "Session expired, please sign in again." });
                    return;
                }

                // 3.5 Ensure we have the latest session (setSession might return old one, getSession returns resolved state)
                const { data: { session } } = await supabase.auth.getSession();

                // CRITICAL: If the session was refreshed, update storage!
                if (session && session.access_token !== supabase_session.access_token) {
                    console.debug("[background] Session refreshed! Updating storage...");
                    await chrome.storage.local.set({ supabase_session: session });
                }

                // 4. Check for existence (Duplicate Check)
                const payload = {
                    // Use the user ID from the potentially refreshed session
                    user_id: session?.user?.id || supabase_session.user.id,
                    ...request.data
                };

                console.debug('[background] Checking for duplicates before insert...');
                const { data: existingData, error: existingError } = await supabase
                    .from('learned_subtitles')
                    .select('id')
                    .eq('user_id', payload.user_id)
                    .eq('movie_url', payload.movie_url)
                    .eq('subtitle_start_ms', payload.subtitle_start_ms)
                    .eq('sentence', payload.sentence)
                    .maybeSingle();

                if (existingData) {
                    console.debug('[background] Subtitle already exists in DB. Skipping insert.');
                    sendResponse({ success: true, message: "Already saved", data: existingData });
                    return;
                }

                if (existingError) {
                    console.warn('[background] Error checking for duplicates:', existingError);
                    // Proceed to insert anyway if check fails? Or fail? 
                    // Let's proceed, as it might just be a connection blip, but logging it is good.
                }

                // 5. Insert Data
                console.debug('[background] Inserting payload:', payload);

                const { data, error } = await supabase
                    .from('learned_subtitles')
                    .insert([payload])
                    .select();

                if (error) {
                    console.error('[background] Database Insert Error:', error);
                    sendResponse({ success: false, error: error.message });
                } else {
                    console.debug('[background] Insert Success:', data);
                    sendResponse({ success: true, data: data });
                }

            } catch (err) {
                console.error('[background] SAVE_SUBTITLE Exception:', err);
                sendResponse({ success: false, error: err.message });
            }
        })();

        return true;
    }

    // NEW: Explicit Refresh Handler for Content Scripts
    if (request.action === "REFRESH_SESSION") {
        console.debug('[background] REFRESH_SESSION requested');
        (async () => {
            try {
                const { createClient } = self.supabase;
                const { SUPABASE_URL, SUPABASE_ANON_KEY } = self.CONFIG;
                const supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY, {
                    auth: { persistSession: false, autoRefreshToken: true, detectSessionInUrl: false }
                });

                const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
                if (!supabase_session) {
                    sendResponse({ success: false, error: "No session stored" });
                    return;
                }

                const { error } = await supabase.auth.setSession({
                    access_token: supabase_session.access_token,
                    refresh_token: supabase_session.refresh_token
                });

                const { data: { session } } = await supabase.auth.getSession();

                if (error || !session) {
                    const errMsg = error?.message || "Refresh failed";

                    // Check for invalid refresh token and clear session if so
                    if (errMsg.includes("Refresh Token Not Found") ||
                        errMsg.includes("Invalid Refresh Token") ||
                        errMsg.includes("refresh_token_not_found")) {
                        console.warn("[background] Invalid refresh token. Clearing session.");
                        await chrome.storage.local.remove(['supabase_session']);
                    }

                    sendResponse({ success: false, error: errMsg });
                } else {
                    // Update storage if changed
                    if (session.access_token !== supabase_session.access_token) {
                        console.debug("[background] REFRESH_SESSION: Updating storage with new token");
                        await chrome.storage.local.set({ supabase_session: session });
                    }
                    console.debug('[background] Session valid/refreshed:', session.user?.id);
                    sendResponse({ success: true, session: session });
                }
            } catch (e) {
                console.error("[background] REFRESH_SESSION Error:", e);
                sendResponse({ success: false, error: e.message });
            }
        })();
        return true;
        return true;
    }

    if (request.action === "GET_NETFLIX_PRELOADED_STATE") {
        (async () => {
            try {
                if (!sender.tab || !sender.tab.id) {
                    sendResponse({ success: false, error: "No tab ID" });
                    return;
                }
                const results = await chrome.scripting.executeScript({
                    target: { tabId: sender.tab.id },
                    world: 'MAIN',
                    func: () => {
                        try {
                            const state = window.__PRELOADED_STATE__;
                            if (state && state.videos) {
                                const videoId = Object.keys(state.videos)[0];
                                return state.videos[videoId]?.title;
                            }
                        } catch (e) { }
                        return null;
                    }
                });
                const title = results && results[0] && results[0].result ? results[0].result : null;
                sendResponse({ success: true, title: title });
            } catch (e) {
                sendResponse({ success: false, error: e.message });
            }
        })();
        return true;
    }

    if (request.action === "GET_NETFLIX_TITLE") {
        console.debug('[background] GET_NETFLIX_TITLE requested');
        (async () => {
            try {
                if (!sender.tab || !sender.tab.id) {
                    sendResponse({ success: false, error: "No tab ID" });
                    return;
                }

                const results = await chrome.scripting.executeScript({
                    target: { tabId: sender.tab.id },
                    world: 'MAIN',
                    func: () => {
                        try {
                            const state = window.netflix?.reactContext?.models;
                            if (!state) return null;

                            for (const key in state) {
                                const model = state[key];
                                if (model?.data?.video?.title) {
                                    return model.data.video.title;
                                }
                            }
                        } catch (e) {
                            console.error("Error accessing Netflix state:", e);
                        }
                        return null;
                    }
                });

                const title = results && results[0] && results[0].result ? results[0].result : null;
                console.debug("[background] Netflix title found:", title);
                sendResponse({ success: true, title: title });

            } catch (e) {
                console.error("[background] GET_NETFLIX_TITLE Error:", e);
                sendResponse({ success: false, error: e.message });
            }
        })();
        return true;
    }
});


