let cachedCaptionTracks = [];
let cachedTranslationLanguages = [];
let currentVideoId = null;
let lastAddedTrack = null;
let currentLearnLang = null;
let currentKnownLang = null;
let hasActiveLearnSubtitles = false;
let hasActiveKnowSubtitles = false;
// let lastTimedTextFetchTime = 0; // Removed in favor of global storage

// Style management
function updateSubtitleStyles(settings) {
    if (typeof LANGUAGES !== 'undefined') {
        // console.debug(`[LingoPause] Loaded ${LANGUAGES.length} languages.`);
    } else {
        console.warn("[LingoPause] LANGUAGES not defined.");
    }

    const root = document.documentElement;

    if (settings.fontSize) {
        const size = settings.fontSize;
        root.style.setProperty('--multisubs-font-size-learn', `${size}px`);
        root.style.setProperty('--multisubs-font-size-roma', `${size * 0.9}px`);
        root.style.setProperty('--multisubs-font-size-know', `${size * 0.7}px`);
        console.debug(`[LingoPause] Font size updated to ${size}px`);
    }

    if (settings.bgOpacity !== undefined) {
        const opacity = settings.bgOpacity / 100;
        root.style.setProperty('--multisubs-bg-color', `rgba(0, 0, 0, ${opacity})`);
        console.debug(`[LingoPause] Background opacity updated to ${opacity}`);
    }

    if (settings.bgMode) {
        const container = document.getElementById('multisubs-subtitle-overlay');
        if (container) {
            if (settings.bgMode === 'container') {
                container.classList.add('multisubs-full-bg');
            } else {
                container.classList.remove('multisubs-full-bg');
            }
        }
        console.debug(`[LingoPause] Background mode updated to ${settings.bgMode}`);
    }

    // Visibility toggles
    const container = document.getElementById('multisubs-subtitle-overlay');
    if (container) {
        if (settings.showKnowLang === false) { // Strict check for false, undefined means show
            container.classList.add('multisubs-hide-know');
        } else if (settings.showKnowLang === true) {
            container.classList.remove('multisubs-hide-know');
        }

        if (settings.showTransliteration === false) {
            container.classList.add('multisubs-hide-roma');
        } else if (settings.showTransliteration === true) {
            container.classList.remove('multisubs-hide-roma');
        }
    }
}

// Initial load of styles
chrome.storage.sync.get(['fontSize', 'bgMode', 'bgOpacity', 'showKnowLang', 'showTransliteration'], (result) => {
    updateSubtitleStyles(result);
});

// Listen for style changes
chrome.storage.onChanged.addListener((changes, area) => {
    if (area === 'sync') {
        const newSettings = {};
        if (changes.fontSize) newSettings.fontSize = changes.fontSize.newValue;
        if (changes.bgMode) newSettings.bgMode = changes.bgMode.newValue;
        if (changes.bgOpacity) newSettings.bgOpacity = changes.bgOpacity.newValue;
        if (changes.showKnowLang) newSettings.showKnowLang = changes.showKnowLang.newValue;
        if (changes.showTransliteration) newSettings.showTransliteration = changes.showTransliteration.newValue;

        if (Object.keys(newSettings).length > 0) {
            updateSubtitleStyles(newSettings);
        }
    }
});

// --- Chrome AI Translation Support ---

async function checkChromeAIAvailability() {
    if (!('translation' in self && 'getAll' in self.translation)) {
        return { available: false, reason: 'api_missing' };
    }
    return { available: true };
}

function showChromeAIInstructions() {
    // Check if we already showed it this session
    if (sessionStorage.getItem('lingopause_ai_instructions_shown')) return;
    sessionStorage.setItem('lingopause_ai_instructions_shown', 'true');

    const modal = document.createElement('div');
    modal.style.cssText = `
        position: fixed;
        top: 0; left: 0; width: 100%; height: 100%;
        background: rgba(0,0,0,0.8);
        z-index: 2147483649;
        display: flex;
        justify-content: center;
        align-items: center;
        font-family: system-ui, sans-serif;
    `;
    modal.innerHTML = `
        <div style="background: #1a1a1a; color: white; padding: 30px; border-radius: 12px; max-width: 600px; line-height: 1.6; border: 1px solid #333;">
            <h2 style="margin-top: 0; color: #4CAF50;">Enable Free AI Translation</h2>
            <p>LingoPause can use Chrome's built-in AI to translate subtitles for free. To enable this experimental feature:</p>
            <ol style="margin: 20px 0; padding-left: 20px;">
                <li>Open a new tab and go to <code style="background: #333; padding: 2px 6px; border-radius: 4px;">chrome://flags</code></li>
                <li>Search for and enable:
                    <ul style="margin: 10px 0;">
                        <li><strong>Enables optimization guide on device</strong> &rarr; <span style="color: cyan;">Enabled BypassPerfRequirement</span></li>
                        <li><strong>Translation API</strong> &rarr; <span style="color: cyan;">Enabled</span></li>
                    </ul>
                </li>
                <li>Click <strong>Relaunch</strong> to restart Chrome.</li>
            </ol>
            <p style="font-size: 0.9em; color: #aaa;">Note: After restarting, Chrome may take a few minutes to download the AI models.</p>
            <button id="lingopause-ai-close" style="background: #4CAF50; color: white; border: none; padding: 10px 20px; border-radius: 6px; cursor: pointer; font-weight: bold; margin-top: 10px;">Got it</button>
        </div>
    `;
    document.body.appendChild(modal);
    document.getElementById('lingopause-ai-close').onclick = () => modal.remove();
}

async function translateVTT(vttContent, sourceLang, targetLang) {
    if (!vttContent) return "";

    console.debug(`[LingoPause] Attempting AI translation from ${sourceLang} to ${targetLang}`);

    try {
        if (!('translation' in self && 'createTranslator' in self.translation)) {
            console.warn("[LingoPause] Chrome Translation API not available.");
            return vttContent; // return original as fallback
        }

        // 1. Create Translator
        // Normalize languages (e.g. "en-US" -> "en")
        const src = sourceLang.split('-')[0];
        const tgt = targetLang.split('-')[0];

        const canTranslate = await translation.canTranslate({
            sourceLanguage: src,
            targetLanguage: tgt,
        });

        if (canTranslate === 'no') {
            console.warn(`[LingoPause] AI Translation unavailable for pair ${src}->${tgt}`);
            showChromeAIInstructions(); // Prompt user if it's missing (could be model not downloaded)
            return vttContent;
        }

        let translator;
        if (canTranslate === 'readily') {
            translator = await translation.createTranslator({ sourceLanguage: src, targetLanguage: tgt });
        } else {
            // 'after-download' -> this might take time
            console.debug("[LingoPause] AI Model needs download...");
            translator = await translation.createTranslator({ sourceLanguage: src, targetLanguage: tgt });
            // Wait for download? The promise above waits.
        }

        // 2. Parse VTT
        // Simple regex-based parsing to extract text blocks
        // VTT Structure:
        // TIME --> TIME
        // Text line 1
        // Text line 2

        const lines = vttContent.split(/\r?\n/);
        let outputLines = [];
        let buffer = [];
        let isHeader = true;

        for (const line of lines) {
            if (isHeader) {
                outputLines.push(line);
                if (line.trim() === "") isHeader = false;
                continue;
            }

            if (line.includes("-->")) {
                // Push buffer (previous text) if exists
                if (buffer.length > 0) {
                    const originalText = buffer.join('\n');
                    try {
                        const translatedText = await translator.translate(originalText);
                        outputLines.push(translatedText);
                    } catch (err) {
                        outputLines.push(originalText); // Fallback to original on error
                        console.error("Translation error:", err);
                    }
                    buffer = [];
                }
                outputLines.push(line); // The timestamp line
            } else if (line.trim() === "") {
                // Blank line separators
                if (buffer.length > 0) {
                    const originalText = buffer.join('\n');
                    try {
                        const translatedText = await translator.translate(originalText);
                        outputLines.push(translatedText);
                    } catch (err) {
                        outputLines.push(originalText);
                    }
                    buffer = [];
                }
                outputLines.push(line);
            } else if (!isNaN(line.trim())) {
                // Index number (optional in VTT but common in SRT)
                outputLines.push(line);
            } else {
                // Actual text content
                buffer.push(line);
            }
        }

        // flush buffer
        if (buffer.length > 0) {
            const originalText = buffer.join('\n');
            try {
                const translatedText = await translator.translate(originalText);
                outputLines.push(translatedText);
            } catch (err) {
                outputLines.push(originalText);
            }
        }

        return outputLines.join('\n');

    } catch (e) {
        console.error("[LingoPause] AI Translation Fatal Error:", e);
        return vttContent;
    }
}

const subtitleContentCache = new Map();

function clearSubtitleContentCache() {
    subtitleContentCache.clear();
    console.debug("[LingoPause] Subtitle content cache cleared.");
}

function setupNetflixSubtitleData(tracks) {
    // console.debug("[LingoPause] Setting up Netflix subtitle data", tracks);
    if (!tracks) return;

    function findFirstUrl(obj) {
        if (typeof obj === 'string' && (obj.startsWith('http'))) return obj;
        if (obj && typeof obj === 'object') {
            for (const key in obj) {
                const res = findFirstUrl(obj[key]);
                if (res) return res;
            }
        }
        return null;
    }

    const incomingTracks = tracks.map(track => {
        // 1. Dehydrated tracks
        if (track.isHydrated === false) {
            return {
                baseUrl: "DEHYDRATED",
                vssId: track.bcp47,
                languageCode: track.bcp47,
                label: { "runs": [{ "text": track.lang }] },
                kind: "standard",
                isNetflix: true,
                isDehydrated: true
            };
        }

        // 2. Processed tracks (Factory output)
        if (track.urls && Array.isArray(track.urls)) {
            return {
                baseUrl: track.urls[0],
                vssId: track.bcp47,
                languageCode: track.bcp47,
                label: { "runs": [{ "text": track.lang }] },
                kind: track.isCaption ? "asr" : "standard",
                isNetflix: true
            };
        }

        // 3. Raw tracks fallback
        const baseUrl = findFirstUrl(track.ttDownloadUrls || track.downloadUrls || track.ttDownloadables || {});

        return {
            baseUrl: baseUrl,
            vssId: track.language,
            languageCode: track.language,
            label: {
                "runs": [{ "text": track.language + (track.isLowIntermediate ? " (SDH)" : "") }]
            },
            kind: track.isCaption ? "asr" : "standard", // Map to internal types
            isNetflix: true
        };
    }).filter(t => t.baseUrl); // Remove tracks with no URL

    // MERGE LOGIC:
    // If we have existing cached tracks, we want to update them with new ones,
    // BUT preserve any "hydrated" (valid URL) tracks if the new one is "DEHYDRATED".
    if (!cachedCaptionTracks || cachedCaptionTracks.length === 0) {
        cachedCaptionTracks = incomingTracks;
    } else {
        // Create a map of existing tracks for fast lookup
        // Key: languageCode + kind (roughly unique)
        const existingMap = new Map();
        cachedCaptionTracks.forEach(t => {
            const key = t.languageCode + "_" + t.kind;
            existingMap.set(key, t);
        });

        cachedCaptionTracks = incomingTracks.map(newTrack => {
            const key = newTrack.languageCode + "_" + newTrack.kind;
            const existing = existingMap.get(key);

            if (existing) {
                // If existing is Hydrated and New is Dehydrated -> KPI EXISTING
                if (existing.baseUrl !== "DEHYDRATED" && newTrack.baseUrl === "DEHYDRATED") {
                    // console.debug(`[LingoPause] Preserving hydrated track for ${newTrack.languageCode} despite incoming dehydration.`);
                    return existing;
                }
            }
            return newTrack;
        });
    }
    // console.debug("[LingoPause] Cached caption tracks:", cachedCaptionTracks);
    cachedTranslationLanguages = []; // Netflix doesn't do "auto-translate" in the same way YouTube does
}

function get_ytplayer_to_body() {
    execJsFileInWebScope('inline-get.js');
}

async function send_ytplayer() {
    // console.debug("[LingoPause] send_ytplayer()");
    try {
        get_ytplayer_to_body();
        await sleep(100);
        //  console.debug("[LingoPause] finished get_ytplayer_to_body()")

        var playerResponse_json = document.body.getAttribute('data-playerResponse');
        // console.debug([LingoPause] playerResponse_json", playerResponse_json);
        for (let i = 0; i < 20 && !playerResponse_json; i++) {
            await sleep(100);
            playerResponse_json = document.body.getAttribute('data-playerResponse');
            // console.debug(`[LingoPause] retry reading body attr ${i + 1}`);
        }
        // console.debug("[LingoPause] finished reading body attr")

        console.debug(document.title, window.location.href);
        let plyRes = null;
        if (playerResponse_json) {
            try {
                plyRes = JSON.parse(playerResponse_json);
                if (plyRes.captions && plyRes.captions.playerCaptionsTracklistRenderer) {
                    cachedCaptionTracks = plyRes.captions.playerCaptionsTracklistRenderer.captionTracks || [];
                    cachedTranslationLanguages = plyRes.captions.playerCaptionsTracklistRenderer.translationLanguages || [];
                } else {
                    cachedCaptionTracks = [];
                    cachedTranslationLanguages = [];
                }
                // autoTriggerCC();
                // autoTriggerCC();
            } catch (e) {
                console.error("Failed to parse playerResponse", e);
                cachedCaptionTracks = [];
                cachedTranslationLanguages = [];
            }
        } else {
            cachedCaptionTracks = [];
            cachedTranslationLanguages = [];
        }

        console.debug(document.title, window.location.href);

        chrome.runtime.sendMessage({
            title: document.title,
            href: window.location.href,
            playerResponse_json: playerResponse_json,
        });
        // console.debug("finished runtime.sendMessage()");

        // console.debug("wait some secs then remove body attr");
        // console.debug("wait some secs then remove body attr");
        sleep(2000).then(() => {
            // console.debug("now remove body attr");
            remove_page_change();
        });

        return plyRes;
    } catch (err) {
        console.debug("!!!! ERROR !!! error when sending ytplayer");
        console.warn(err);
        return null;
    }
}


// --- Check if captions are active ---
const captionsActive = () => {
    const video = document.querySelector("video");
    if (!video || !video.textTracks) return false;
    return [...video.textTracks].some(
        t => t.kind === "subtitles" || t.kind === "captions"
    );
};

// --- Activate captions ---
const activateCaptions = async () => {
    if (location.hostname.includes('netflix.com')) return true; // Netflix handles this differently
    console.debug("[LingoPause] Activating Captions");

    // Wait until ad is not showing (div.ad-showing is not present)
    const adWaitStart = Date.now();
    const AD_WAIT_TIMEOUT = 10000; // 10 seconds max wait

    while (document.querySelector("div.ad-showing") && Date.now() - adWaitStart < AD_WAIT_TIMEOUT) {
        console.debug("[LingoPause] Waiting for ad to finish...");
        await new Promise(r => setTimeout(r, 500));
    }

    if (document.querySelector("div.ad-showing")) {
        // console.warn("[LingoPause] Ad still showing after timeout, proceeding anyway");
    } else {
        // console.debug("[LingoPause] No ad detected, proceeding with caption activation");
    }

    const ccButton = document.querySelector(".ytp-subtitles-button");
    if (!ccButton) return false;

    // Only activate if not already active
    if (!captionsActive()) {
        ccButton.click();
        await new Promise(r => setTimeout(r, 300));
        ccButton.click(); // second click to ensure UI toggled
    }

    // Wait for captions to actually load
    const start = Date.now();
    while (!captionsActive() && Date.now() - start < 2000) {
        await new Promise(r => setTimeout(r, 200));
    }

    return captionsActive();
};

// --- Deactivate captions ---
const deactivateCaptions = async () => {
    if (location.hostname.includes('netflix.com')) return true;
    // console.debug("[LingoPause] Deactivating Captions");
    const ccButton = document.querySelector(".ytp-subtitles-button");
    if (!ccButton) return false;

    // Only turn off if currently active
    if (captionsActive()) {
        ccButton.click();
        await new Promise(r => setTimeout(r, 300));
    }

    return !captionsActive();
};

function get_subtitle_url_by_label(label) {
    if (!cachedCaptionTracks || cachedCaptionTracks.length === 0) return "";

    // Find the track that matches the user's selected language label
    const track = cachedCaptionTracks.find(t => t.name.simpleText === label);
    if (!track) return "";

    let url = track.baseUrl + "&fmt=vtt";
    // c=WEB is usually needed, pot is for authentication if available
    url += `&c=WEB`;

    // We should ideally use the pot token if we have one, but for now we'll rely on the base URL
    return url;
}




function getLanguageName(code) {
    if (!code) return "Unknown";
    // Using shared LANGUAGES array
    if (typeof LANGUAGES !== 'undefined') {
        const lang = LANGUAGES.find(l => l.code.toLowerCase() === code.toLowerCase());
        if (lang) return lang.name;
    }
    return code; // Fallback to code if name not found or LANGUAGES undefined
}

function normalizeLanguageCode(lang) {
    if (!lang) return "";
    const trimmed = lang.trim();

    // First check if it's already a valid code in our dictionary
    if (typeof LANGUAGES !== 'undefined') {
        // Direct code match?
        if (LANGUAGES.some(l => l.code.toLowerCase() === trimmed.toLowerCase())) {
            return trimmed.toLowerCase();
        }

        // Name match?
        const langByName = LANGUAGES.find(l => l.name.toLowerCase() === trimmed.toLowerCase());
        if (langByName) {
            return langByName.code;
        }
    }

    // Fallback if not found 
    return trimmed.toLowerCase();
}

function get_subtitle_url_by_language_code(langCode, pot) {
    const normalizedRequestedCode = normalizeLanguageCode(langCode);
    // console.debug(`[LingoPause] get_subtitle_url_by_language_code called for: "${langCode}" (normalized: "${normalizedRequestedCode}")`);
    if (!cachedCaptionTracks || cachedCaptionTracks.length === 0) {
        // console.warn("[LingoPause] No caption tracks cached yet!");
        return "";
    }

    // console.debug("[LingoPause] Total cached tracks:", cachedCaptionTracks.length);
    cachedCaptionTracks.forEach((t, i) => {
        const urlStr = typeof t.baseUrl === 'string' ? t.baseUrl : "NOT_A_STRING";
        // console.debug(`[LingoPause] Track[${i}]: langCode=${t.languageCode}, isNetflix=${t.isNetflix}, baseUrl=${urlStr.substring(0, 50)}...`);
    });

    let url = "";
    // 1. Try to find a direct track for this language
    // Match exactly or startsWith (to handle en-US vs en)
    const tracks = cachedCaptionTracks.filter(t => {
        const trackCode = normalizeLanguageCode(t.languageCode);
        return trackCode === normalizedRequestedCode ||
            trackCode.startsWith(normalizedRequestedCode + "-") ||
            normalizedRequestedCode.startsWith(trackCode + "-");
    }).sort((a, b) => {
        // Prioritize hydrated tracks (baseUrl !== "DEHYDRATED")
        const aHydrated = a.baseUrl !== "DEHYDRATED";
        const bHydrated = b.baseUrl !== "DEHYDRATED";
        if (aHydrated && !bHydrated) return -1;
        if (!aHydrated && bHydrated) return 1;
        return 0;
    });

    // console.debug(`[LingoPause] Matching tracks for "${langCode}":`, tracks.length);

    if (tracks.length > 0) {
        // Prefer manual tracks over auto-generated (kind !== 'asr')
        const manualTrack = tracks.find(t => t.kind !== 'asr');
        const track = manualTrack || tracks[0];
        url = track.baseUrl;
        if (!track.isNetflix) {
            url += "&fmt=vtt";
        }
        // console.debug("[LingoPause] Selected direct track URL:", url, "isDehydrated:", track.baseUrl === "DEHYDRATED");
    }
    // 2. Fallback: Check if we can translate an existing track to this language
    else {
        // console.debug("[LingoPause] No direct track found, checking translation languages...");
        const canTranslate = cachedTranslationLanguages.some(l => l.languageCode === langCode);
        // console.debug("[LingoPause] Can Translate:", canTranslate);

        if (canTranslate) {
            const englishTrack = cachedCaptionTracks.find(t => t.languageCode === 'en' && t.kind !== 'asr');
            const baseTrack = englishTrack || cachedCaptionTracks[0];
            // console.debug("[LingoPause] Base Track for Translation:", baseTrack?.languageCode);

            if (baseTrack) {
                url = baseTrack.baseUrl;
                if (!baseTrack.isNetflix) {
                    url += "&fmt=vtt";
                    url += `&tlang=${langCode}`;
                }
                // console.debug("[LingoPause] Generated translated URL:", url);
            }
        }
    }

    if (url && !cachedCaptionTracks.some(t => t.baseUrl === url && t.isNetflix)) {
        if (!url.includes("&c=WEB")) url += `&c=WEB`;
        if (pot && !url.includes("&pot=")) {
            url += `&pot=${pot}`;
        }
    }

    console.debug(`[LingoPause] Final URL for "${langCode}":`, url || "NONE");
    return url;
}

async function fetchWithRetry(url, {
    retries = 2,
    retryDelayMs = 300,
    label = "subtitle"
} = {}) {
    let lastText = "";
    let currentRetryDelay = retryDelayMs;

    for (let attempt = 0; attempt <= retries; attempt++) {
        console.debug(
            `[LingoPause] Fetching ${label} (attempt ${attempt + 1}/${retries + 1}): ${url}`
        );

        const resp = await fetch(url);

        if (!resp.ok) {
            console.warn(
                `[LingoPause] ${label} fetch failed. Status: ${resp.status} ${resp.statusText}`
            );

            if (resp.status === 429) {
                console.warn(`[LingoPause] ${label} 429 Too Many Requests. Stopping retries.`);
                throw new Error("429_TOO_MANY_REQUESTS");
            }

            if (attempt < retries) {
                console.warn(
                    `[LingoPause] ${label} failed (${resp.status}), retrying in ${currentRetryDelay}ms…`
                );
                await new Promise(r => setTimeout(r, currentRetryDelay));
                currentRetryDelay *= 2; // Exponential backoff
                continue;
            }

            return null; // hard failure after retries exhausted
        }

        const text = await resp.text();
        lastText = text;

        // ✅ RETRY CONDITION: length === 0
        if (text.length > 0) {
            return text; // success
        }

        if (attempt < retries) {
            console.warn(
                `[LingoPause] ${label} returned empty body (length=0), retrying in ${retryDelayMs}ms…`
            );
            await new Promise(r => setTimeout(r, retryDelayMs));
        }
    }

    // fetched successfully but empty after all retries
    return lastText; // ""
}

// Removed lastErrorTimeout as we handle individual messages
// Shared debounce state for player errors
let lastPlayerErrorMessage = null;
let lastPlayerErrorTime = 0;

function showInPlayerError(message) {
    console.debug("[LingoPause] showInPlayerError", message);
    // Add prefix if not present
    if (!message.startsWith("[LingoPause]")) {
        message = `[LingoPause] ${message}`;
    }

    // Debounce check (5 seconds)
    if (message === lastPlayerErrorMessage && Date.now() - lastPlayerErrorTime < 5000) {
        console.debug("[LingoPause] Suppressing duplicate player error:", message);
        return;
    }
    lastPlayerErrorMessage = message;
    lastPlayerErrorTime = Date.now();

    console.error(`[LingoPause] Player Error: ${message}`);

    // Find the correct video container
    let videoContainer = null;
    const subContainer = document.getElementById('multisubs-subtitle-overlay');
    if (subContainer && subContainer.parentNode) {
        videoContainer = subContainer.parentNode;
    } else {
        // Fallback logic
        let video = document.querySelector('video');
        if (location.hostname.includes('netflix.com') && typeof NetflixAPI !== 'undefined') {
            videoContainer = NetflixAPI.getPlayerContainer();
        }
        if (!videoContainer) {
            videoContainer = document.querySelector('div[data-uia="video-canvas"]') ||
                document.querySelector('.html5-video-player') ||
                document.querySelector('.watch-video') ||
                document.querySelector('.videostream') ||
                video?.parentElement ||
                document.body;
        }
    }

    if (!videoContainer) {
        console.warn("[LingoPause] showInPlayerError: No video container found!");
        return;
    }
    console.debug("[LingoPause] showInPlayerError using container:", videoContainer);

    let errorContainer = document.getElementById('multisubs-error-display');
    if (!errorContainer) {
        errorContainer = document.createElement('div');
        errorContainer.id = 'multisubs-error-display';
        errorContainer.style.cssText = `
            position: absolute;
            top: 10%;
            right: 5%;
            z-index: 2147483648;
            max-width: 400px;
            pointer-events: none;
            display: flex;
            flex-direction: column;
            gap: 10px;
        `;
        videoContainer.appendChild(errorContainer);
    }

    // Create the individual message element
    const messageEl = document.createElement('div');
    messageEl.style.cssText = `
        background: rgba(220, 53, 69, 0.95); 
        color: white; 
        padding: 15px 20px; 
        border-radius: 8px; 
        font-family: system-ui, sans-serif; 
        font-size: 16px; 
        font-weight: 500;
        box-shadow: 0 4px 15px rgba(0,0,0,0.6);
        border: 1px solid #ff4444;
        display: flex;
        align-items: flex-start;
        gap: 10px;
        opacity: 0;
        transition: opacity 0.5s ease, transform 0.5s ease;
        transform: translateY(-20px);
    `;

    messageEl.innerHTML = `
        <span style="font-size: 20px;">⚠️</span>
        <span>${message}</span>
    `;

    // Append to container
    errorContainer.appendChild(messageEl);

    // Trigger animation
    // Force reflow
    messageEl.offsetHeight;
    messageEl.style.opacity = '1';
    messageEl.style.transform = 'translateY(0)';

    // Remove after 5 seconds
    setTimeout(() => {
        if (messageEl && messageEl.parentNode) {
            messageEl.style.opacity = '0';
            messageEl.style.transform = 'translateY(-20px)';

            // Remove from DOM after fade out
            setTimeout(() => {
                if (messageEl.parentNode) {
                    messageEl.remove();
                    // Clean up container if empty
                    if (errorContainer.childNodes.length === 0) {
                        errorContainer.remove();
                    }
                }
            }, 500);
        }
    }, 5000);
}

// --- Persistent Caching Helpers ---

async function cleanUpOldSubtitles() {
    return new Promise((resolve) => {
        chrome.storage.local.get(null, (items) => {
            if (chrome.runtime.lastError) {
                console.warn("[LingoPause] Cleanup fetch error:", chrome.runtime.lastError);
                resolve();
                return;
            }

            const now = Date.now();
            const THREE_HOURS_MS = 3 * 60 * 60 * 1000;
            const keysToRemove = [];

            for (const [key, value] of Object.entries(items)) {
                // Heuristic: Subtitle keys are URLs. Settings are short strings usually.
                // Or checking value structure.
                if (key.includes('http') || (value && (value.text || typeof value === 'string'))) {
                    // Check if it's a subtitle entry
                    // 1. New format: { text, timestamp }
                    if (value && value.timestamp) {
                        if (now - value.timestamp > THREE_HOURS_MS) {
                            keysToRemove.push(key);
                        }
                    }
                    // 2. Legacy format (string) or un-timestamped
                    // Ideally we migrate or delete.
                    // User said "remove caches more than 3 hours ago".
                    // If we can't tell the time, we should probably assume it's old if it's a legacy item from a previous session?
                    // But to be safe, let's only delete if we are SURE it's old (timestamp check).
                    // OR, since we are moving to a new system, maybe we just delete all legacy strings to clean up?
                    // Let's stick to the requested logic: only delete if > 3h.
                    // If no timestamp, we ignore it for now to avoid deleting fresh legacy items (though they shouldn't exist after reload).
                }
            }

            if (keysToRemove.length > 0) {
                console.debug(`[LingoPause] Cleaning up ${keysToRemove.length} old subtitle files...`);
                chrome.storage.local.remove(keysToRemove, () => {
                    if (chrome.runtime.lastError) {
                        console.warn("[LingoPause] Cleanup remove error:", chrome.runtime.lastError);
                    }
                    resolve();
                });
            } else {
                resolve();
            }
        });
    });
}
async function getSubtitleFromStorage(url) {
    if (!url) return null;
    return new Promise((resolve) => {
        chrome.storage.local.get([url], (result) => {
            if (chrome.runtime.lastError) {
                console.warn("[LingoPause] Storage get error:", chrome.runtime.lastError);
                resolve(null);
            } else {
                const item = result[url];
                if (!item) {
                    resolve(null);
                } else if (typeof item === 'string') {
                    // Legacy format
                    resolve(item);
                } else if (item.text) {
                    // New format
                    resolve(item.text);
                } else {
                    resolve(null);
                }
            }
        });
    });
}

async function saveSubtitleToStorage(url, content) {
    if (!url || !content) return;

    // Trigger cleanup before saving (fire and forget or await?)
    // Await to ensure space is made if needed, but don't block too long.
    // Actually, chrome.storage.local is async, so better to await.
    await cleanUpOldSubtitles();

    return new Promise((resolve) => {
        const data = {};
        // Save with timestamp
        data[url] = {
            text: content,
            timestamp: Date.now()
        };

        chrome.storage.local.set(data, () => {
            if (chrome.runtime.lastError) {
                console.warn("[LingoPause] Storage set error:", chrome.runtime.lastError.message);
                // If quota exceeded, we could try a more aggressive cleanup here if we wanted.
            }
            resolve();
        });
    });
}

async function fetch_and_merge_subtitles(learnLang, knowLang, retryCount = 0) {
    const { isActive } = await chrome.storage.sync.get(['isActive']);
    if (!isActive) return;

    if (location.hostname.includes("youtube.com")) {
        const AD_WAIT_TIMEOUT = 10000;
        const adWaitStart = Date.now();
        while (document.querySelector("div.ad-showing") && Date.now() - adWaitStart < AD_WAIT_TIMEOUT) {
            console.debug("[LingoPause] Waiting for ad to finish before fetching...");
            await new Promise(r => setTimeout(r, 500));
        }

        // Feature: Force subtitle toggle on initial load to ensure hydration/POT
        if (retryCount === 0) {
            console.debug("[LingoPause] Pre-emptive caption toggle to ensure hydration...");
            await activateCaptions();
            await sleep(CONFIG.DELAY_BETWEEN_CAPTION_ACTIVATION);
            await deactivateCaptions();
        }
    }

    currentLearnLang = learnLang;
    currentKnownLang = knowLang;
    // console.debug("[LingoPause] fetch_and_merge_subtitles, retryCount:", retryCount);
    const MAX_RETRIES = 2;

    // Rate Limit Check
    let initialVideoId = null;
    if (location.hostname.includes("youtube.com")) {
        initialVideoId = new URLSearchParams(window.location.search).get('v');
    }
    const now = Date.now();

    // helper to get last fetch time from storage
    const getLastFetchTime = () => new Promise(resolve => {
        chrome.storage.local.get(['lastTimedTextFetchTime'], (result) => {
            resolve(result.lastTimedTextFetchTime || 0);
        });
    });

    // helper to set last fetch time to storage
    const setLastFetchTime = (time) => new Promise(resolve => {
        chrome.storage.local.set({ lastTimedTextFetchTime: time }, resolve);
    });

    const lastFetchTime = await getLastFetchTime();
    const timeSinceLastFetch = now - lastFetchTime;

    // Use config value or default to 20 seconds
    const MIN_FETCH_INTERVAL = (typeof CONFIG !== 'undefined' && CONFIG.MIN_FETCH_INTERVAL) ? CONFIG.MIN_FETCH_INTERVAL : 20000;

    if (timeSinceLastFetch < MIN_FETCH_INTERVAL && location.hostname.includes("youtube.com")) {
        // Update immediately to reserve the slot (accounting for wait)
        const newNextFetchTime = now + (MIN_FETCH_INTERVAL - timeSinceLastFetch);
        // We set the time AS IF we just fetched, to block others immediately
        // BUT actually we add the wait time to "reserve" it
        // Actually, simpler logic:
        // We want to wait, then fetch. But to prevent race conditions from other tabs,
        // we should probably update the storage immediately to say "I am fetching very soon".
        // Let's set it to 'now' effectively pushing the next available slot.
        // However, since we wait, we should strictly set it to when we PLAN to fetch?
        // Let's just update it to now so other tabs see it as "just fetched" and back off.
        await setLastFetchTime(now);

        const waitTime = MIN_FETCH_INTERVAL - timeSinceLastFetch;
        console.debug(`[LingoPause] Global rate limit active. Waiting ${waitTime}ms before fetching...`);
        await sleep(waitTime);

        // Check if video changed during wait
        if (initialVideoId) {
            const currentVideoId = new URLSearchParams(window.location.search).get('v');
            if (currentVideoId && initialVideoId !== currentVideoId) {
                console.debug(`[LingoPause] Video changed during rate limit wait. Aborting fetch.`);
                return;
            }
        }
        console.debug("[LingoPause] Resuming fetch after rate limit wait...");
    } else if (location.hostname.includes("youtube.com")) {
        // No wait, but reserve slot
        console.debug("[LingoPause] Global rate limit passed. No wait.");
        await setLastFetchTime(now);
    }

    let pot = "";
    if (location.hostname.includes("youtube.com")) {
        try {
            pot = await chrome.runtime.sendMessage({ action: "get-pot" });
            console.debug("[LingoPause] retrieved pot from background:", pot ? (pot.substring(0, 10) + "...") : "missing");
        } catch (e) {
            console.warn("[LingoPause] failed to get pot from background", e);
        }
    }

    let learnUrl = "";
    let knowUrl = "";

    // On YouTube, we strictly require POT to attempt a fetch.
    if (!location.hostname.includes("youtube.com") || pot) {
        learnUrl = get_subtitle_url_by_language_code(learnLang, pot) || "";
        knowUrl = get_subtitle_url_by_language_code(knowLang, pot) || "";
    } else {
        console.warn("[LingoPause] Missing POT token for YouTube. Skipping fetch to trigger fallback/retry.");
    }

    // If on Netflix and a language is missing, show a clear error in the player
    // ONLY check if user is logged in
    const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
    if (location.hostname.includes('netflix.com') && supabase_session?.access_token) {
        // Check for DEHYDRATED tracks
        // Only show this error on the first attempt to avoid duplicates during retries
        if (retryCount === 0 && (learnUrl === "DEHYDRATED" || knowUrl === "DEHYDRATED")) {
            const dehydratedLang = learnUrl === "DEHYDRATED" ? learnLang : knowLang;
            const langName = getLanguageName(dehydratedLang);
            const msg = `The ${langName} subtitles are available but not enabled. Please switch "${langName}" Netflix subtitles on and off to enable them.`;
            showInPlayerError(msg);
            // We can still try to process the other language if available, or just return.
            // If we return, we stop everything. Let's see if we should fetch the other one.
            // For now, let's treat it as a hard error for that track.
        }

        let missingMessages = [];
        // Only report missing if NOT dehydrated (handled above) and NOT found
        if (!learnUrl && learnLang) {
            missingMessages.push(`${getLanguageName(learnLang)} subtitles`);
        }
        if (!knowUrl && knowLang) {
            missingMessages.push(`${getLanguageName(knowLang)} subtitles`);
        }

        if (missingMessages.length > 0) {
            const finalMessage = missingMessages.join(' and ') + (missingMessages.length > 1 ? ' are ' : ' is ') + 'not found for this netflix video.';

            // Wait for video to be somewhat ready before showing error
            // This prevents the error from flashing before the video even starts
            const maxWait = 10; // 10 seconds
            let waited = 0;
            const checkAndShow = () => {
                const video = document.querySelector('video');
                if (video && video.readyState >= 1) { // HAVE_METADATA
                    showInPlayerError(finalMessage);
                } else if (waited < maxWait) {
                    waited++;
                    setTimeout(checkAndShow, 1000);
                } else {
                    // Fallback if video takes too long or isn't found
                    showInPlayerError(finalMessage);
                }
            };
            checkAndShow();
        }
    }

    if (!learnUrl && !knowUrl) {
        console.warn("No subtitle URLs found for selected languages");
        return;
    }

    let learnVtt = "";
    let knowVtt = "";

    try {
        // --- Original Fetch Logic ---
        if (learnUrl && learnUrl !== "DEHYDRATED") {
            if (subtitleContentCache.has(learnUrl)) {
                console.debug("[LingoPause] Using in-memory cached content for learnUrl:", learnUrl);
                learnVtt = subtitleContentCache.get(learnUrl);
            } else {
                // Check persistent storage
                const localContent = await getSubtitleFromStorage(learnUrl);
                if (localContent) {
                    console.debug("[LingoPause] Using persistent cached content for learnUrl:", learnUrl);
                    learnVtt = localContent;
                    subtitleContentCache.set(learnUrl, learnVtt); // Populate L1
                } else {
                    console.debug(`[LingoPause] Fetching learnVtt from: ${learnUrl}`);
                    let rawLearn = await fetchWithRetry(learnUrl, {
                        retries: 0,
                        retryDelayMs: 800,
                        label: "learnVtt"
                    });
                    console.debug(`[LingoPause] learnVtt fetch result: ${rawLearn ? "Success (length: " + rawLearn.length + ")" : "Failed/Empty"}`);

                    // Process format if needed
                    if (typeof rawLearn === 'string' && (rawLearn.trim().startsWith('<?xml') || rawLearn.trim().startsWith('<tt'))) {
                        console.debug("[LingoPause] Detected Netflix TTML format for learnLang");
                        rawLearn = convertTTMLToVTT(rawLearn);
                    } else if (typeof rawLearn === 'string' && rawLearn.length > 0 && !rawLearn.trim().startsWith('WEBVTT')) {
                        console.debug("[LingoPause] learnVtt is not TTML/XML and not WEBVTT. Possible image-based subtitle? First 100 chars:", rawLearn.substring(0, 100));
                    }

                    if (typeof rawLearn === 'string' && rawLearn.length > 0) {
                        learnVtt = rawLearn;
                        subtitleContentCache.set(learnUrl, learnVtt);
                        await saveSubtitleToStorage(learnUrl, learnVtt);
                    }
                }
            }
        }

        if (knowUrl && knowUrl !== "DEHYDRATED") {
            if (subtitleContentCache.has(knowUrl)) {
                console.debug("[LingoPause] Using in-memory cached content for knowUrl:", knowUrl);
                knowVtt = subtitleContentCache.get(knowUrl);
            } else {
                // Check persistent storage
                const localContent = await getSubtitleFromStorage(knowUrl);
                if (localContent) {
                    console.debug("[LingoPause] Using persistent cached content for knowUrl:", knowUrl);
                    knowVtt = localContent;
                    subtitleContentCache.set(knowUrl, knowVtt); // Populate L1
                } else {
                    if (location.hostname.includes("youtube.com") && learnUrl) {
                        console.debug("[LingoPause] Waiting 2s between subtitle fetches...");
                        const delay = (typeof CONSTANTS !== 'undefined' && CONSTANTS.SUBTITLE_FETCH_DELAY) ? CONSTANTS.SUBTITLE_FETCH_DELAY : 3000;
                        await sleep(delay);
                    }
                    console.debug(`[LingoPause] Fetching knowVtt from: ${knowUrl}`);
                    let rawKnow = await fetchWithRetry(knowUrl, {
                        retries: 0,
                        retryDelayMs: 800,
                        label: "knowVtt"
                    });
                    console.debug(`[LingoPause] knowVtt fetch result: ${rawKnow ? "Success (length: " + rawKnow.length + ")" : "Failed/Empty"}`);

                    // Process format if needed
                    if (typeof rawKnow === 'string' && (rawKnow.trim().startsWith('<?xml') || rawKnow.trim().startsWith('<tt'))) {
                        console.debug("[LingoPause] Detected Netflix TTML format for knowLang");
                        rawKnow = convertTTMLToVTT(rawKnow);
                    } else if (typeof rawKnow === 'string' && rawKnow.length > 0 && !rawKnow.trim().startsWith('WEBVTT')) {
                        console.debug("[LingoPause] knowVtt is not TTML/XML and not WEBVTT. Possible image-based subtitle? First 100 chars:", rawKnow.substring(0, 100));
                    }

                    if (typeof rawKnow === 'string' && rawKnow.length > 0) {
                        knowVtt = rawKnow;
                        subtitleContentCache.set(knowUrl, knowVtt);
                        await saveSubtitleToStorage(knowUrl, knowVtt);
                    }
                }
            }
        }

        // NOTE: Only doing this for Netflix for now as YouTube has its own thing.
        if (location.hostname.includes('netflix.com')) {
            const aiStatus = await checkChromeAIAvailability();
            console.debug(`[LingoPause] Chrome AI Status:`, aiStatus);

            if (aiStatus.available) {
                if (!learnVtt && learnLang) {
                    console.debug(`[LingoPause] Evaluation for AI Translate (Learn): knowVtt=${!!knowVtt}, learnLang=${learnLang}`);
                    // Need Learn, have Know?
                    if (knowVtt) {
                        console.debug(`[LingoPause] Missing learnVtt (${learnLang}). Attempting AI translation from knowVtt (${knowLang}).`);
                        console.debug(`[LingoPause] Translate input length: ${knowVtt.length}`);
                        learnVtt = await translateVTT(knowVtt, knowLang, learnLang);
                        console.debug(`[LingoPause] Translate output length: ${learnVtt ? learnVtt.length : 0}`);
                    } else {
                        // Have neither? Try to fetch English track specifically for translation source
                        // TODO: Ideally we'd find the best available source language
                        console.debug(`[LingoPause] Missing learnVtt (${learnLang}). No knowVtt to translate from.`);
                    }
                }

                if (!knowVtt && knowLang) {
                    console.debug(`[LingoPause] Evaluation for AI Translate (Know): learnVtt=${!!learnVtt}, knowLang=${knowLang}`);
                    // Need Know, have Learn?
                    if (learnVtt) {
                        console.debug(`[LingoPause] Missing knowVtt (${knowLang}). Attempting AI translation from learnVtt (${learnLang}).`);
                        console.debug(`[LingoPause] Translate input length: ${learnVtt.length}`);
                        knowVtt = await translateVTT(learnVtt, learnLang, knowLang);
                        console.debug(`[LingoPause] Translate output length: ${knowVtt ? knowVtt.length : 0}`);
                    }
                }
            } else {
                console.debug(`[LingoPause] Chrome AI not available, skipping translation fallback. Reason: ${aiStatus.reason || 'unknown'}`);
            }
        }
    } catch (e) {
        if (e.message === "429_TOO_MANY_REQUESTS") {
            console.warn("[LingoPause] Aborting fetch_and_merge_subtitles due to 429.");
            showInPlayerError("[LingoPause] Unable to get subtitles from the platform right now (429). Please wait a while and try again later.");
            return;
        }
        console.error("[LingoPause] Exception during fetch:", e);
    }

    const hasLearn = learnVtt && learnVtt.trim().length > 0;
    const hasKnow = knowVtt && knowVtt.trim().length > 0;

    if (!hasLearn && !hasKnow) {
        console.warn("Both subtitle fetches failed or were empty");
        if (retryCount < MAX_RETRIES) {
            console.debug("[LingoPause] Activating and deactivating captions to try to hydrate subtitles");
            await activateCaptions();
            await sleep(CONFIG.DELAY_BETWEEN_CAPTION_ACTIVATION); // Wait for CC to activate and pot to be captured
            await deactivateCaptions();

            // Retry the function
            return await fetch_and_merge_subtitles(learnLang, knowLang, retryCount + 1);
        }
        showInPlayerError("[LingoPause] Failed to load subtitles. Please try turning the captions on and off in the video, or refreshing the page.");
        return;
    }

    // Update global state
    hasActiveLearnSubtitles = !!(learnUrl && learnVtt && learnVtt.trim().length > 0);
    hasActiveKnowSubtitles = !!(knowUrl && knowVtt && knowVtt.trim().length > 0);
    console.debug("[LingoPause] hasActiveLearnSubtitles:", hasActiveLearnSubtitles);
    console.debug("[LingoPause] hasActiveKnowSubtitles:", hasActiveKnowSubtitles);

    // Ensure tracker is enabled now that we have subtitles
    if (typeof usageTracker !== 'undefined') {
        usageTracker.enable();
    }

    const mergedVtt = await mergeSubtitles(learnVtt, knowVtt, learnLang);
    await display_merged_sub(mergedVtt);
}

async function mergeSubtitles(learnVtt, knowVtt, learnLang) {
    // console.debug("[LingoPause] mergeSubtitles called. learnVtt length:", learnVtt?.length, "knowVtt length:", knowVtt?.length);
    if (!learnVtt) return knowVtt;
    if (!knowVtt) return learnVtt;

    const parseTime = (timeStr) => {
        // Convert "00:00:01.500" to milliseconds
        const [hours, minutes, rest] = timeStr.split(':');
        const [seconds, ms] = rest.split('.');
        return (
            parseInt(hours) * 3600000 +
            parseInt(minutes) * 60000 +
            parseInt(seconds) * 1000 +
            parseInt(ms)
        );
    };

    const formatTime = (ms) => {
        const hours = Math.floor(ms / 3600000);
        const minutes = Math.floor((ms % 3600000) / 60000);
        const seconds = Math.floor((ms % 60000) / 1000);
        const milliseconds = ms % 1000;

        return `${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}.${String(milliseconds).padStart(3, '0')}`;
    };

    const normalizeCueTime = (timeLine) => {
        return timeLine.replace(
            /^(\d{2}:\d{2}:\d{2}\.\d{3}\s*-->\s*\d{2}:\d{2}:\d{2}\.\d{3}).*$/,
            "$1"
        );
    };

    const isTimestampLine = (line) =>
        /^\d{2}:\d{2}:\d{2}\.\d{3}\s*-->\s*\d{2}:\d{2}:\d{2}\.\d{3}/.test(line);

    const cleanInlineTiming = (text) => {
        return text
            .replace(/<\d{2}:\d{2}:\d{2}\.\d{3}>/g, "")
            .replace(/\b\d{1,3}%\b/g, "")
            .replace(/\s{2,}/g, " ")
            .trim();
    };

    const parseVTT = (vtt) => {
        const lines = vtt.split(/\r?\n/);
        const cues = [];
        let currentCue = null;

        for (let i = 0; i < lines.length; i++) {
            const line = lines[i].trim();
            if (line.includes("-->")) {
                if (currentCue) cues.push(currentCue);

                const normalized = normalizeCueTime(line);
                const [startStr, endStr] = normalized.split('-->').map(s => s.trim());

                currentCue = {
                    time: normalized,
                    startMs: parseTime(startStr),
                    endMs: parseTime(endStr),
                    text: ""
                };
            } else if (
                currentCue &&
                line &&
                line !== "WEBVTT" &&
                !line.startsWith("STYLE") &&
                !line.startsWith("::cue") &&
                !isTimestampLine(line) &&
                !/^\d+$/.test(line)
            ) {
                currentCue.text += (currentCue.text ? "\n" : "") + line;
            }
        }
        if (currentCue) cues.push(currentCue);
        return cues;
    };

    const learnCues = parseVTT(learnVtt);
    const knowCues = parseVTT(knowVtt);

    // Function to find overlapping know cues for a given learn cue
    const findOverlappingKnowCues = (learnCue) => {
        const overlapping = [];
        for (const knowCue of knowCues) {
            // Check if there's any overlap
            const overlapStart = Math.max(learnCue.startMs, knowCue.startMs);
            const overlapEnd = Math.min(learnCue.endMs, knowCue.endMs);

            if (overlapStart < overlapEnd) {
                // There's overlap - calculate how much
                const overlapDuration = overlapEnd - overlapStart;
                const learnDuration = learnCue.endMs - learnCue.startMs;
                const overlapPercentage = (overlapDuration / learnDuration) * 100;

                overlapping.push({
                    cue: knowCue,
                    overlapPercentage: overlapPercentage
                });
            }
        }

        // Sort by overlap percentage (descending)
        overlapping.sort((a, b) => b.overlapPercentage - a.overlapPercentage);
        return overlapping;
    };

    const stripVttTags = (text) =>
        text.replace(/<\/?c(\.[^>]*)?>/g, "");

    let merged = "WEBVTT\n\n";
    const usedKnowCues = new Set();

    // Process each learn cue
    for (const learnCue of learnCues) {
        merged += learnCue.time + " align:center\n";

        const cleanedLearnText = stripVttTags(cleanInlineTiming(learnCue.text));
        let finalText = await processTextWithRomanization(cleanedLearnText, learnLang);

        // Find overlapping know cues
        const overlapping = findOverlappingKnowCues(learnCue);

        // Always add learn text first (on top)
        merged += finalText + "\n";

        // Add overlapping know text(s) below
        if (overlapping.length > 0) {
            // Use the most overlapping know cue(s)
            // You can adjust the threshold - currently using 30% overlap minimum
            const significantOverlaps = overlapping.filter(o => o.overlapPercentage > 30);

            if (significantOverlaps.length > 0) {
                const knowTexts = significantOverlaps.map(o => {
                    usedKnowCues.add(o.cue);
                    return stripVttTags(cleanInlineTiming(o.cue.text));
                }).filter(t => t); // Remove empty texts

                if (knowTexts.length > 0) {
                    merged += "<c.know-lang>" + knowTexts.join(' ') + "</c>\n";
                }
            }
        }

        merged += "\n";
    }

    // Add any remaining know cues that weren't used (no significant overlap with any learn cue)
    for (const knowCue of knowCues) {
        if (!usedKnowCues.has(knowCue)) {
            const cleanedText = stripVttTags(cleanInlineTiming(knowCue.text));
            if (cleanedText) {
                merged += knowCue.time + " align:center\n";
                merged += "<c.know-lang>" + cleanedText + "</c>\n\n";
            }
        }
    }

    return merged;
}

// async function mergeSubtitles(learnVtt, knowVtt, learnLang) {
//     if (!learnVtt) return knowVtt;
//     if (!knowVtt) return learnVtt;

//     const normalizeCueTime = (timeLine) => {
//         return timeLine.replace(
//             /^(\d{2}:\d{2}:\d{2}\.\d{3}\s*-->\s*\d{2}:\d{2}:\d{2}\.\d{3}).*$/,
//             "$1"
//         );
//     };

//     const isTimestampLine = (line) =>
//         /^\d{2}:\d{2}:\d{2}\.\d{3}\s*-->\s*\d{2}:\d{2}:\d{2}\.\d{3}/.test(line);

//     const cleanInlineTiming = (text) => {
//         return text
//             .replace(/<\d{2}:\d{2}:\d{2}\.\d{3}>/g, "")
//             .replace(/\b\d{1,3}%\b/g, "")
//             .replace(/\s{2,}/g, " ")
//             .trim();
//     };

//     const parseVTT = (vtt) => {
//         const lines = vtt.split(/\r?\n/);
//         const cues = [];
//         let currentCue = null;

//         for (let i = 0; i < lines.length; i++) {
//             const line = lines[i].trim();
//             if (line.includes("-->")) {
//                 if (currentCue) cues.push(currentCue);
//                 currentCue = {
//                     time: normalizeCueTime(line),
//                     text: ""
//                 };
//             } else if (
//                 currentCue &&
//                 line &&
//                 line !== "WEBVTT" &&
//                 !line.startsWith("STYLE") &&
//                 !line.startsWith("::cue") &&
//                 !isTimestampLine(line) &&
//                 !/^\d+$/.test(line)
//             ) {
//                 currentCue.text += (currentCue.text ? "\n" : "") + line;
//             }
//         }
//         if (currentCue) cues.push(currentCue);
//         return cues;
//     };

//     const learnCues = parseVTT(learnVtt);
//     const knowCues = parseVTT(knowVtt);

//     let merged = "WEBVTT\n\n";

//     const knowMap = new Map();
//     knowCues.forEach(c => knowMap.set(c.time, c.text));

//     const isChinese = learnLang && learnLang.startsWith('zh');
//     const isKorean = learnLang && learnLang.startsWith('ko');

//     for (const lc of learnCues) {
//         merged += lc.time + "\n";

//         const kt = knowMap.get(lc.time);

//         const cleanedLearnText = cleanInlineTiming(lc.text);
//         let finalText = await processTextWithRomanization(cleanedLearnText, learnLang);

//         // Always add learn text first
//         merged += finalText + "\n";

//         if (kt) {
//             // Add a line break and wrap the known language text in a class for styling
//             merged += "<c.know-lang>" + kt + "</c>\n\n";
//             knowMap.delete(lc.time); // Mark as used
//         } else {
//             merged += "\n"; // Ensure there's an extra line if no know text is available
//         }
//     }

//     // Add remaining know cues that didn't match
//     knowMap.forEach((text, time) => {
//         merged += time + "\n<c.know-lang>" + text + "</c>\n\n";
//     });

//     return merged;
// }


async function processTextWithRomanization(text, learnLang) {
    const isChinese = learnLang && learnLang.startsWith('zh');
    const isKorean = learnLang && learnLang.startsWith('ko');
    const isThai = learnLang && learnLang.startsWith('th');
    const isJapanese = learnLang && learnLang.startsWith('ja');

    if (!isChinese && !isKorean && !isThai && !isJapanese) return text;

    const lines = text.split(/\r?\n/);
    let finalText = "";

    // Use for buffer to properly await
    for (let i = 0; i < lines.length; i++) {
        const line = lines[i];
        finalText += (i > 0 ? "\n" : "") + line;

        let romanization = "";
        if (isChinese) {
            romanization = getPinyinForText(line, learnLang);
        } else if (isKorean) {
            romanization = getRomanizedKorean(line);
        } else if (isThai) {
            romanization = getRomanizedThai(line);
        } else if (isJapanese) {
            romanization = await getRomanizedJapanese(line);
        }

        if (romanization) {
            finalText += `\n<c.pinyin>${romanization}</c>`;
        }
    }
    return finalText;
}

function getPinyinForText(text, lang) {
    if (!lang || !lang.startsWith('zh')) return "";

    // Optional: still check for chinese characters to avoid generating pinyin for non-chinese parts mixed in?
    if (!/[\u4e00-\u9fa5]/.test(text)) return "";

    return getPinyin(text) || "";
}

let kuroshiroObj = null;
let kuroshiroInitPromise = null;

async function getRomanizedJapanese(text) {
    // Check if Kuroshiro is loaded in page context
    const isLoaded = document.documentElement.getAttribute('data-kuroshiro-loaded') === 'true';
    if (!isLoaded) {
        // Wait a bit for scripts to load
        await new Promise(resolve => setTimeout(resolve, 1000));
        if (document.documentElement.getAttribute('data-kuroshiro-loaded') !== 'true') {
            return null;
        }
    }

    try {
        // Use background script to execute conversion in page context (avoids CSP issues)
        const response = await new Promise((resolve) => {
            const dictPath = chrome.runtime.getURL("./kuroshiro/");
            chrome.runtime.sendMessage(
                {
                    action: "convert-japanese",
                    text: text,
                    dictPath: dictPath
                },
                (response) => {
                    if (chrome.runtime.lastError) {
                        resolve({ success: false, error: chrome.runtime.lastError.message });
                    } else {
                        resolve(response || { success: false, error: "No response" });
                    }
                }
            );
        });

        if (response.success) {
            return response.result;
        } else {
            console.warn("Japanese romanization failed:", response.error);
            return null;
        }
    } catch (e) {
        console.warn("Japanese romanization failed:", e);
        return null;
    }
}

function getRomanizedKorean(text) {
    if (typeof Aromanize === 'undefined') return null;
    try {
        return Aromanize.romanize(text);
    } catch (e) {
        console.warn("Korean romanization failed:", e);
        return null;
    }
}

function getRomanizedThai(text) {
    if (typeof romanizeThai === 'undefined') return null;
    try {
        return romanizeThai(text);
    } catch (e) {
        console.warn("Thai romanization failed:", e);
        return null;
    }
}

function getPinyin(text) {
    if (typeof pinyinPro === 'undefined') return null;
    try {
        return pinyinPro.pinyin(text, { toneType: 'symbol', type: 'string' });
    } catch (e) {
        console.warn("Pinyin generation failed:", e);
        return null;
    }
}

async function getRomanizationForToken(token, lang) {
    if (!lang) return "";
    if (lang.startsWith('ja')) return await getRomanizedJapanese(token);
    if (lang.startsWith('zh')) return getPinyinForText(token, lang);
    if (lang.startsWith('ko')) return getRomanizedKorean(token);
    if (lang.startsWith('th')) return getRomanizedThai(token);
    return "";
}

// Parse VTT content into cue objects with timing
function parseVTTContent(vttContent) {
    // console.debug("[LingoPause] parseVTTContent called. Input length:", vttContent?.length);
    const cues = [];
    if (!vttContent) return cues;
    const lines = vttContent.split(/\r?\n/);
    let i = 0;

    while (i < lines.length) {
        const line = lines[i].trim();

        // Look for timestamp lines (HH:MM:SS.mmm --> HH:MM:SS.mmm)
        if (line.includes('-->')) {
            const timeMatch = line.match(/(\d{2}:\d{2}:\d{2}\.\d{3})\s*-->\s*(\d{2}:\d{2}:\d{2}\.\d{3})/);
            if (timeMatch) {
                const startTime = timeMatch[1];
                const endTime = timeMatch[2];

                // Convert time to seconds
                const timeToSeconds = (timeStr) => {
                    const [hours, minutes, seconds] = timeStr.split(':');
                    const [secs, ms] = seconds.split('.');
                    return parseInt(hours) * 3600 + parseInt(minutes) * 60 + parseInt(secs) + parseInt(ms) / 1000;
                };

                const startSeconds = timeToSeconds(startTime);
                const endSeconds = timeToSeconds(endTime);

                // Collect all text lines until next timestamp or empty line
                i++;
                let text = '';
                while (i < lines.length && lines[i].trim() && !lines[i].includes('-->')) {
                    text += (text ? '\n' : '') + lines[i].trim();
                    i++;
                }

                cues.push({
                    start: startSeconds,
                    end: endSeconds,
                    text: text
                });
            }
        }
        i++;
    }

    // console.debug("[LingoPause] Parsed", cues.length, "cues from VTT");
    // console.debug(`[LingoPause] parseVTTContent finished. Found ${cues.length} cues.`);
    return cues;
}

function normalizeVTTCueLines(text) {
    if (!text) return "";

    return text
        // opening tags → own line
        .replace(/(<c(?:\.[^>]*)?>)/g, "\n$1\n")
        // closing tags → own line
        .replace(/(<\/c>)/g, "\n$1\n")
        // clean up extra blank lines
        .replace(/\n{2,}/g, "\n")
        .trim();
}

function convertVTTToHTML(text) {
    if (!text) return '';
    text = normalizeVTTCueLines(text);
    //console.debug("[LingoPause] convertVTTToHTML input:", text);

    // First, let's handle multi-line blocks by collecting lines between tags
    const lines = text.split('\n');
    let html = '';
    let currentBlock = null; // 'know-lang', 'pinyin', or null
    let blockContent = [];

    for (let line of lines) {
        line = line.trim();
        if (!line) continue;

        // Check for opening tags
        if (line === '<c.know-lang>') {
            currentBlock = 'know-lang';
            blockContent = [];
            continue;
        } else if (line === '<c.pinyin>') {
            currentBlock = 'pinyin';
            blockContent = [];
            continue;
        } else if (line === '<c>' || line.match(/^<c(\.[^>]*)?>$/)) {
            // Generic opening tag, treat as learn
            currentBlock = 'learn';
            blockContent = [];
            continue;
        }

        // Check for closing tag
        if (line === '</c>') {
            // Output the accumulated block
            if (currentBlock && blockContent.length > 0) {
                const content = blockContent.join(' ').trim();
                if (content) {
                    if (currentBlock === 'know-lang') {
                        html += `<div class="multisubs-know">${content}</div>`;
                    } else if (currentBlock === 'pinyin') {
                        html += `<div class="multisubs-roma">${content}</div>`;
                    } else {
                        html += `<div class="multisubs-learn">${content}</div>`;
                    }
                }
            }
            currentBlock = null;
            blockContent = [];
            continue;
        }

        // If we're inside a block, accumulate content
        if (currentBlock) {
            // Remove any inline VTT tags from the line
            const cleanLine = line.replace(/<\/?c(\.[^>]*)?>/g, '').trim();
            if (cleanLine) {
                blockContent.push(cleanLine);
            }
        } else {
            // Not inside a block - treat as regular learn language text
            const content = line.replace(/<\/?c(\.[^>]*)?>/g, '').trim();
            if (content) {
                html += `<div class="multisubs-learn">${content}</div>`;
            }
        }
    }

    //console.debug("[LingoPause] convertVTTToHTML output:", html);
    return html;
}

// Store cues and current subtitle container
let subtitleCues = [];
let subtitleContainer = null;
let renderLoopId = null; // Changed from videoTimeUpdateListener

// Helper to ensure subtitle container is attached to DOM
function ensureSubtitleContainer(video) {
    if (subtitleContainer && subtitleContainer.parentNode && document.contains(subtitleContainer)) {
        return subtitleContainer;
    }

    // console.debug("[LingoPause] Subtitle container missing or detached, finding home...");

    let videoContainer = null;
    if (location.hostname.includes('netflix.com')) {
        if (typeof NetflixAPI !== 'undefined') {
            videoContainer = NetflixAPI.getPlayerContainer();
        }
    }

    if (!videoContainer) {
        videoContainer = document.querySelector('div[data-uia="video-canvas"]') ||
            document.querySelector('.html5-video-player') ||
            document.querySelector('.watch-video') ||
            document.querySelector('.videostream') ||
            video?.parentElement ||
            document.body;
    }

    if (!videoContainer) {
        console.error("[LingoPause] Could not find any container for subtitles!");
        return null;
    }

    if (!subtitleContainer) {
        subtitleContainer = document.createElement('div');
        subtitleContainer.id = 'multisubs-subtitle-overlay';
    }

    // Set/Update styles
    subtitleContainer.style.cssText = `
        position: absolute;
        bottom: 15%; /* Higher than YouTube's default to avoid Netflix controls */
        left: 50%;
        transform: translateX(-50%);
        width: 80%;
        max-width: 1200px;
        text-align: center;
        pointer-events: none;
        z-index: 2147483647;
        display: none;
    `;

    if (subtitleContainer.parentElement !== videoContainer) {
        videoContainer.appendChild(subtitleContainer);
        // console.debug("[LingoPause] Subtitle overlay attached/re-attached to:", videoContainer.className || videoContainer.id || videoContainer.tagName);
    }

    // Apply background mode class on creation/re-attachment
    // Apply styling classes on creation/re-attachment
    chrome.storage.sync.get(['bgMode', 'showKnowLang', 'showTransliteration'], (result) => {
        if (result.bgMode === 'container') {
            subtitleContainer.classList.add('multisubs-full-bg');
        } else {
            subtitleContainer.classList.remove('multisubs-full-bg');
        }

        // Toggle visibility classes based on settings (default true if undefined)
        // If showKnowLang is FALSE, we ADD the hide class.
        if (result.showKnowLang === false) {
            subtitleContainer.classList.add('multisubs-hide-know');
        } else {
            subtitleContainer.classList.remove('multisubs-hide-know');
        }

        if (result.showTransliteration === false) {
            subtitleContainer.classList.add('multisubs-hide-roma');
        } else {
            subtitleContainer.classList.remove('multisubs-hide-roma');
        }
    });

    return subtitleContainer;
}

// Modify display_merged_sub to use HTML overlay instead of VTT track
async function display_merged_sub(vttContent) {
    // console.debug("[LingoPause] ======= DISPLAY_MERGED_SUB CALLED =======");

    const { isActive } = await chrome.storage.sync.get(['isActive']);
    if (!isActive) return;

    // Authentication Check
    const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
    if (!supabase_session?.access_token) {
        console.warn("[LingoPause] Access denied: User not authenticated.");
        // showErrorIndicator("Please log in to use LingoPause");
        return;
    }

    // Usage Allowance Check
    const isAllowed = await usageTracker.checkAllowance();
    if (!isAllowed) {
        console.warn("[LingoPause] Usage allowance check failed.");
        return;
    }

    // Initial cleanup
    removeDualSubtitles();
    injectStyles();

    // Find video
    let video = null;
    if (location.hostname.includes('netflix.com')) {
        if (typeof NetflixAPI !== 'undefined') {
            try {
                video = await NetflixAPI.waitForVideoElement(10000);
                // console.debug("[LingoPause] Found Netflix video via NetflixAPI: Success");
            } catch (e) {
                // console.warn("[LingoPause] NetflixAPI.waitForVideoElement timed out:", e);
            }
        }
    }

    if (!video) {
        const videos = document.getElementsByTagName("video");
        if (videos.length === 0) {
            // console.warn("[LingoPause] No video found immediately, waiting briefly...");
            for (let i = 0; i < 10; i++) {
                await new Promise(r => setTimeout(r, 500));
                const retryVideos = document.getElementsByTagName("video");
                if (retryVideos.length > 0) {
                    video = retryVideos[0];
                    break;
                }
            }
        } else {
            video = videos[0];
        }
    }

    if (!video) {
        console.error("[LingoPause] CRITICAL: No video found after waiting!");
        return;
    }

    // Parse VTT content into cues
    subtitleCues = parseVTTContent(vttContent);
    // console.debug(`[LingoPause] Parsed ${subtitleCues.length} cues from VTT`);
    if (subtitleCues.length > 0) {
        // console.debug(`[LingoPause] Subtitle Timing Range: ${subtitleCues[0].start.toFixed(2)}s to ${subtitleCues[subtitleCues.length - 1].end.toFixed(2)}s`);
    }

    // Start High-Performance Renderer Loop
    startRendererLoop(video);

    // console.debug("[LingoPause] ======= HTML SUBTITLE DISPLAY STARTED =======");
}

function startRendererLoop(video) {
    if (renderLoopId) {
        cancelAnimationFrame(renderLoopId);
    }

    let lastTime = -1;
    let lastLogTime = 0;

    const renderFrame = () => {
        // 1. Self-Healing check (ensure container exists)
        const container = ensureSubtitleContainer(video);
        if (!container) {
            renderLoopId = requestAnimationFrame(renderFrame);
            return;
        }

        const currentTime = video.currentTime;

        // Throttled log (every 5 seconds)
        if (Date.now() - lastLogTime > 5000) {
            lastLogTime = Date.now();
        }

        // Only update if time has actually changed significantly (e.g. > 10ms)
        // or if we need a periodic check
        if (Math.abs(currentTime - lastTime) > 0.01) {
            lastTime = currentTime;

            // Check for ads (YouTube specific)
            const adShowing = document.querySelector('.ad-showing') || document.querySelector('.ad-interrupting');
            if (adShowing) {
                if (container.style.display !== 'none') {
                    container.style.display = 'none';
                }
                renderLoopId = requestAnimationFrame(renderFrame);
                return;
            }

            // Find active cues
            const activeCues = subtitleCues.filter(cue =>
                currentTime >= cue.start && currentTime <= cue.end
            );

            if (activeCues.length > 0) {
                // console.debug("There is active cues");
                const combinedText = activeCues.map(cue => cue.text).join('\n');
                //console.debug("combinedText", combinedText);
                const html = convertVTTToHTML(combinedText);

                // Clear and render new content if changed
                if (container.innerHTML !== html) {
                    container.innerHTML = html;
                }

                if (container.style.display !== 'block') {
                    container.style.display = 'block';
                }
            } else {
                if (container.style.display !== 'none') {
                    container.style.display = 'none';
                    container.innerHTML = ''; // Keep it clean
                }
            }
        }

        renderLoopId = requestAnimationFrame(renderFrame);
    };

    renderLoopId = requestAnimationFrame(renderFrame);
}

function stopRendererLoop() {
    if (renderLoopId) {
        cancelAnimationFrame(renderLoopId);
        renderLoopId = null;
    }
}


function injectStyles() {
    if (document.getElementById("multisubs-style")) return;

    const style = document.createElement("style");
    style.id = "multisubs-style";
    style.textContent = `
        /* HTML Subtitle Overlay Styles */
        #multisubs-subtitle-overlay .multisubs-learn {
            font-size: var(--multisubs-font-size-learn, 24px);
            text-shadow: 2px 2px 4px #000;
            padding: 0.2em 0.6em;
            background-color: var(--multisubs-bg-color, rgba(0, 0, 0, 0.8));
            color: #ffffff;
            line-height: 1.05;
            font-family: system-ui, sans-serif;
            display: inline-block;
            margin: 2px auto;
            width: fit-content;
        }
        
        #multisubs-subtitle-overlay .multisubs-roma {
            font-size: var(--multisubs-font-size-roma, 21.6px);
            color: #aaddff;
            text-shadow: 1px 1px 2px #000;
            line-height: 1.05;
            font-family: system-ui, sans-serif;
            display: block;
            margin: 0 auto;
            padding: 0.1em 0.4em;
            background-color: var(--multisubs-bg-color, rgba(0, 0, 0, 0.7));
            width: fit-content;
        }
        
        #multisubs-subtitle-overlay .multisubs-know {
            font-size: var(--multisubs-font-size-know, 16.8px);
            color: #cccccc;
            text-shadow: 1px 1px 2px #000;
            line-height: 1.05;
            font-family: system-ui, sans-serif;
            display: block;
            margin: 4px auto 0 auto;
            padding: 0.2em 0.5em;
            background-color: var(--multisubs-bg-color, rgba(0, 0, 0, 0.7));
            width: fit-content;
        }

        /* Container background mode */
        #multisubs-subtitle-overlay.multisubs-full-bg {
            background-color: var(--multisubs-bg-color, rgba(0, 0, 0, 0.8));
            padding: 15px;
            border-radius: 8px;
        }
        #multisubs-subtitle-overlay.multisubs-full-bg .multisubs-learn,
        #multisubs-subtitle-overlay.multisubs-full-bg .multisubs-roma,
        #multisubs-subtitle-overlay.multisubs-full-bg .multisubs-know {
            background-color: transparent !important;
            box-shadow: none !important;
        }
        
        #multisubs-loading-indicator {
            position: absolute;
            bottom: 60px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 10px 20px;
            border-radius: 10px;
            z-index: 60;
            display: flex;
            align-items: center;
            gap: 10px;
            font-family: Roboto, Arial, sans-serif;
            font-size: 14px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.5);
            transition: opacity 0.3s ease-in-out;
        }
        .multisubs-spinner {
            width: 16px;
            height: 16px;
            border: 2px solid #ffffff;
            border-top: 2px solid transparent;
            border-radius: 50%;
            animation: multisubs-spin 1s linear infinite;
        }
        @keyframes multisubs-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        #multisubs-study-spinner {
            position: fixed;
            top: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.7);
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            font-family: system-ui, sans-serif;
            font-size: 13px;
            z-index: 2147483647;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        .multisubs-spinner-small {
            width: 14px;
            height: 14px;
            border: 2px solid #ffffff;
            border-top: 2px solid transparent;
            border-radius: 50%;
            border-top: 2px solid transparent;
            border-radius: 50%;
            animation: multisubs-spin 1s linear infinite;
        }

        /* Hides elements ONLY in the playback overlay, not the pause screen */
        #multisubs-subtitle-overlay.multisubs-hide-know .multisubs-know { display: none !important; }
        #multisubs-subtitle-overlay.multisubs-hide-roma .multisubs-roma { display: none !important; }
    `;
    document.head.appendChild(style);
}

async function showLoadingIndicator() {
    const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
    if (!supabase_session?.access_token) return;
    console.debug("[LingoPause] showLoadingIndicator");
    // Don't show if ad is playing (YouTube)
    if (document.querySelector('.ad-showing')) return;

    injectStyles();
    if (document.getElementById('multisubs-loading-indicator')) return;

    const div = document.createElement('div');
    div.id = 'multisubs-loading-indicator';
    div.innerHTML = `
        <span>LingoPause subtitles loading...</span>
        <div class="multisubs-spinner"></div>
    `;
    document.body.appendChild(div);
}

function hideLoadingIndicator() {
    console.debug("[LingoPause] hideLoadingIndicator");
    const el = document.getElementById('multisubs-loading-indicator');
    if (el) el.remove();
}

let lastErrorMessage = null;
let lastErrorTime = 0;

function showErrorIndicator(message, videoElement = null) {
    // Debounce check
    if (message === lastErrorMessage && Date.now() - lastErrorTime < 5000) {
        return;
    }
    lastErrorMessage = message;
    lastErrorTime = Date.now();

    hideLoadingIndicator(); // Ensure loading indicator is gone

    if (document.getElementById('multisubs-error-indicator')) return;

    injectStyles(); // Ensure styles are present

    // Try to find the video player container to position relative to it
    let videoContainer = null;

    // 1. Prefer passed video element's parent (most reliable)
    if (videoElement && videoElement.parentNode) {
        // For YouTube, parent is usually .html5-video-player.
        // For Netflix, it might be deeper, so we stick to hostname checks below for specifics if needed.
        videoContainer = videoElement.parentNode;
    }

    // 2. Hostname specific overrides
    if (location.hostname.includes('netflix.com')) {
        const netflixCanvas = document.querySelector('div[data-uia="video-canvas"]');
        if (netflixCanvas) videoContainer = netflixCanvas;
    }

    // 3. Fallbacks
    if (!videoContainer) {
        videoContainer = document.querySelector('.html5-video-player') ||
            document.querySelector('.watch-video') ||
            document.querySelector('video')?.parentElement ||
            document.body;
    }

    // console.debug("[LingoPause] showErrorIndicator attaching to:", videoContainer.className || videoContainer.tagName);
    console.debug(`[LingoPause] Showing error indicator: "${message}" inside ${videoContainer.className || videoContainer.tagName}`);

    const div = document.createElement('div');
    div.id = 'multisubs-error-indicator';
    // Ensure [LingoPause] prefix
    if (!message.startsWith('[LingoPause]')) {
        message = `[LingoPause] ${message}`;
    }
    div.innerHTML = `<span>${message}</span>`;

    // Inline style overrides for error specific look (red background)
    // Position top-right as requested, now that we are correctly inside the video container
    div.style.cssText = `
        position: absolute !important;
        top: 20px !important;
        right: 20px !important;
        bottom: auto !important;
        left: auto !important;
        transform: none !important;
        background: rgba(200, 0, 0, 0.95);
        color: white;
        padding: 12px 24px;
        border-radius: 8px;
        z-index: 2147483647;
        display: flex;
        align-items: center;
        font-family: Roboto, Arial, sans-serif;
        font-size: 15px;
        font-weight: 500;
        box-shadow: 0 4px 12px rgba(0,0,0,0.5);
        border: 1px solid rgba(255,255,255,0.2);
        visibility: visible !important;
        opacity: 1 !important;
        pointer-events: auto;
    `;

    videoContainer.appendChild(div);

    // Auto-hide after 5 seconds
    setTimeout(() => {
        if (div.parentNode) {
            div.remove();
        }
    }, 5000);
}

// NUCLEAR OPTION: Remove everything subtitle-related
function nukeAllSubtitles() {
    // console.debug("[LingoPause] ☢️ NUKING ALL SUBTITLES ☢️");

    // 1. Remove all video elements and recreate them
    const videos = document.getElementsByTagName("video");
    // console.debug("[LingoPause] Found", videos.length, "video elements to nuke");

    for (let i = 0; i < videos.length; i++) {
        const video = videos[i];

        // CRITICAL: Clear all cues from text tracks BEFORE removing
        if (video.textTracks) {
            for (let j = 0; j < video.textTracks.length; j++) {
                const track = video.textTracks[j];
                // console.debug("[LingoPause] Track", j, "mode:", track.mode, "cues:", track.cues?.length);

                // Remove all cues
                if (track.cues) {
                    while (track.cues.length > 0) {
                        track.removeCue(track.cues[0]);
                    }
                    // console.debug("[LingoPause] Removed all cues from track", j);
                }

                // Disable the track
                track.mode = 'disabled';
            }
        }

        // Remove ALL track elements
        const allTracks = video.querySelectorAll("track");
        // console.debug("[LingoPause] Removing", allTracks.length, "track DOM elements from video", i);
        allTracks.forEach(t => t.remove());
    }

    // 2. Clear tracks from DOM but KEEP cached track metadata 
    // (so we can re-add them if user toggles extension)
    lastAddedTrack = null;

    // console.debug("[LingoPause] ☢️ NUKE COMPLETE ☢️");
}

/* =========================================================================
   MOVIE METADATA CACHING
   ========================================================================= */
let cachedMetadata = null;

if (location.hostname.includes('netflix.com') && typeof NetflixAPI !== 'undefined' && NetflixAPI.observeNetflixTitle) {
    // console.debug("[LingoPause] Initializing Netflix Title Observer...");

    let lastTitle = null;

    NetflixAPI.observeNetflixTitle((rawTitle) => {
        if (!rawTitle) return;

        const title = rawTitle.trim().toLowerCase();

        if (
            title === lastTitle ||
            title.startsWith("netflix")
        ) return;

        lastTitle = title; // 🔒 lock immediately

        (async () => {
            // console.debug("[LingoPause] Netflix Title changed to:", rawTitle);

            const info = await NetflixAPI.getCurrentVideoInfo();

            cachedMetadata = {
                movieId: info.videoId,
                movieTitle: info.title,
                movieType: 'netflix',
                url: window.location.href
            };

            // console.debug("[LingoPause] Updated cached metadata:", cachedMetadata);
        })();
    });
}
// Ensure we have initial metadata on load/setup
const ensureMetadata = async () => {
    // If we already have valid cached data for the current URL, return it
    if (cachedMetadata && cachedMetadata.url === window.location.href) {
        return cachedMetadata;
    }

    const url = window.location.href;
    let movieId = "";
    let movieTitle = document.title;
    const movieType = location.hostname.includes('netflix') ? 'netflix' : 'youtube';

    if (movieType === 'youtube') {
        const urlObj = new URL(url);
        movieId = urlObj.searchParams.get("v");
        movieTitle = movieTitle.replace(/^\(\d+\)\s*/, "").replace(" - YouTube", "");
    } else if (movieType === 'netflix') {
        // Use the centralized NetflixAPI
        const info = await NetflixAPI.getCurrentVideoInfo();
        movieId = info.videoId;
        movieTitle = info.title;
    }

    cachedMetadata = { movieId, movieTitle, movieType, url };
    // console.debug("[LingoPause] Initialized metadata cache:", cachedMetadata);
    return cachedMetadata;
};

// Modify removeDualSubtitles to be more thorough
function removeDualSubtitles() {

    // console.debug("[LingoPause] ======= REMOVING DUAL SUBTITLES =======");

    // 1. Robust removal: Check DOM directly by ID
    const existingOverlay = document.getElementById('multisubs-subtitle-overlay');
    if (existingOverlay) {
        existingOverlay.remove();
        // console.debug("[LingoPause] Removed existing overlay via DOM ID query.");
    }

    // Capture specific Netflix error display too
    const errorDisplay = document.getElementById('multisubs-error-indicator');
    if (errorDisplay) {
        errorDisplay.remove();
    }

    const loadingIndicator = document.getElementById('multisubs-loading-indicator');
    if (loadingIndicator) {
        loadingIndicator.remove();
    }

    // 2. Clean up variable reference
    if (subtitleContainer) {
        // If it's the same node we just removed, this does nothing, which is fine.
        // If it's a different node (detached), we remove it to be sure.
        subtitleContainer.remove();
        subtitleContainer = null;
    }

    // Remove high-perf renderer loop
    stopRendererLoop();

    // Clear subtitle cues (current display state)
    subtitleCues = [];
    // KEEP cachedCaptionTracks so we can re-enable subtitles without waiting for manifest
    // lastAddedTrack = null;

    // console.debug("[LingoPause] ======= REMOVAL COMPLETE =======");
}

function handleOverlayKeydown(e) {
    if (e.code === 'Space') {
        // Allow native behavior (unpause) by NOT preventing default/propagation
        togglePauseOverlay(false);
    }
}

async function togglePauseOverlay(shouldResume = true) {
    const { isActive } = await chrome.storage.sync.get(['isActive']);
    if (!isActive) return;

    if (!hasActiveLearnSubtitles) {
        // console.debug("[LingoPause] Smart Pause ignored: No active learn subtitles.");
        // Optional: Show a toast? For now, just ignore.
        return;
    }

    const overlayId = "multisubs-overlay";
    const video = document.querySelector("video");
    let overlay = document.getElementById(overlayId);

    /* ================= REMOVE OVERLAY ================= */
    if (overlay) {
        document.removeEventListener('keydown', handleOverlayKeydown, true);
        overlay.remove();
        if (video && shouldResume) video.play();
        return;
    }

    /* ================= CHECK FOR CONTENT ================= */
    // Use parsed subtitle cues instead of VTT track
    const currentTime = video?.currentTime || 0;
    const activeCues = (typeof subtitleCues !== 'undefined' ? subtitleCues : []).filter(cue =>
        currentTime >= cue.start && currentTime <= cue.end
    );

    // console.debug("Active cues at time", currentTime, ":", activeCues);

    if (!activeCues || !activeCues.length) {
        return;
    }

    /* ================= CREATE OVERLAY ================= */
    if (video) video.pause();

    overlay = document.createElement("div");
    overlay.id = overlayId;
    overlay.style.cssText = `
        position: fixed;
        inset: 0;
        background: black;
        z-index: 2147483647;
        display: flex;
        justify-content: center;
        align-items: center;
        cursor: pointer;
    `;

    const container = document.createElement("div");
    container.style.cssText = `
        max-width: 90vw;
        padding: 32px;
        text-align: center;
        color: white;
        display: flex;
        flex-direction: column;
        align-items: center;
    `;

    // --- AUTO-SAVE LOGIC ---
    try {
        const fullSentence = activeCues.map(c => c.text).join('\n')
            .replace(/<\/?c(\.[^>]*)?>/g, "") // Strip active tags
            .replace(/\s+/g, ' ')
            .trim();

        if (fullSentence && currentLearnLang) {
            const payload = {
                learn_language: currentLearnLang,
                know_language: currentKnownLang,
                sentence: fullSentence,
                // translated_sentence: we could try to extract "know" blocks but it's tricky to map perfectly
                // For now just save the raw data or maybe try to extract known?

                movie_title: document.title,
                movie_url: window.location.href,
                movie_type: location.hostname.includes('netflix') ? 'netflix' : 'youtube',

                subtitle_start_ms: Math.floor(activeCues[0].start * 1000),
                subtitle_end_ms: Math.floor(activeCues[activeCues.length - 1].end * 1000),

                // We can save the raw annotations if available from `learnBlocks` later?
                // For now, let's just save the sentence to "learned_subtitles"
            };

            // Extract known language text if possible
            // Re-use logic from below loop or just do a quick collection
            // Let's keep it simple for now and rely on the background to just confirm save

            // console.debug("[LingoPause] Auto-saving subtitle...", payload);
            if (activeCues && activeCues.length > 1) chrome.runtime.sendMessage({
                action: "SAVE_SUBTITLE",
                data: payload
            }, (response) => {
                if (response && response.success) {
                    // Show small toast
                    const toast = document.createElement('div');
                    toast.innerText = "Saved to DB ✓";
                    toast.style.cssText = `
                        position: absolute;
                        top: 20px;
                        left: 50%;
                        transform: translateX(-50%);
                        background: rgba(40, 167, 69, 0.9);
                        color: white;
                        padding: 5px 15px;
                        border-radius: 20px;
                        font-size: 12px;
                        font-family: system-ui;
                        opacity: 0;
                        transition: opacity 0.5s;
                    `;
                    overlay.appendChild(toast);
                    // Force reflow
                    toast.offsetHeight;
                    toast.style.opacity = '1';
                    setTimeout(() => toast.style.opacity = '0', 2000);
                } else {
                    console.warn("[LingoPause] Save failed:", response?.error);
                }
            });
        }
    } catch (e) {
        console.error("[LingoPause] Auto-save error:", e);
    }
    // -----------------------

    let html = "";
    let learnBlocks = []; // To track which blocks need annotation
    let knowBlocks = [];  // To track translation blocks
    let romaBlocks = [];  // To track romanization blocks

    activeCues.forEach((cue, cueIdx) => {
        let text = cue.text || "";

        /* ---------- NORMALIZE YOUTUBE VTT ---------- */
        text = text
            .replace(/(<\/c(\.[^>]*)?>)/g, "\n$1\n")      // close tag → newline
            .replace(/(<c\.[^>]+>)/g, "\n$1\n")           // open tag → newline
            .replace(/\n{2,}/g, "\n")
            .trim();

        const lines = text.split("\n");
        const blocks = [];
        let currentType = null;
        let buffer = [];

        const flush = () => {
            if (currentType && buffer.length) {
                blocks.push({
                    type: currentType,
                    text: buffer.join("<br>").trim()
                });
            }
            buffer = [];
            currentType = null;
        };

        /* ---------- STATEFUL, ORDER-PRESERVING PARSE ---------- */
        for (let raw of lines) {
            let line = raw.trim();
            if (!line) continue;

            if (line.startsWith("<c.pinyin>")) {
                flush();
                currentType = "roma";
                line = line.replace("<c.pinyin>", "").trim();
                if (line) buffer.push(line);
                continue;
            }

            if (line.startsWith("<c.know-lang>")) {
                flush();
                currentType = "know";
                line = line.replace("<c.know-lang>", "").trim();
                if (line) buffer.push(line);
                continue;
            }

            if (line === "</c>") {
                flush();
                continue;
            }

            if (currentType) {
                buffer.push(line);
            } else {
                blocks.push({
                    type: "learn",
                    text: line
                });
            }
        }
        flush();

        /* ---------- RENDER IN EXACT ORDER ---------- */
        html += `<div style="margin-bottom: 24px;">`;

        blocks.forEach((block, blockIdx) => {
            if (!block.text) return;

            if (block.type === "learn") {
                const blockId = `study-learn-${cueIdx}-${blockIdx}`;
                html += `
                    <div id="${blockId}" style="
                        font-size: 5rem;
                        font-family: 'Noto Sans', 'Noto Sans CJK', sans-serif;
                        line-height: 1.25;
                        margin-top: 8px;
                    ">${block.text}</div>
                `;
                learnBlocks.push({ id: blockId, text: block.text });
            }

            if (block.type === "roma") {
                if (!currentLearnLang) {
                    html += `
                        <div style="
                            font-size: 3.2rem;
                            color: #9fd3ff;
                            font-family: 'Noto Sans', 'Noto Sans CJK', sans-serif;
                            margin-top: 0px;
                        ">${block.text}</div>
                    `;
                }
                romaBlocks.push(block.text);
            }

            if (block.type === "know") {
                html += `
                    <div style="
                        font-size: 3rem;
                        color: #bbb;
                        font-family: Roboto, Arial, sans-serif;
                        margin-top: 20px;
                    ">${block.text}</div>
                `;
                knowBlocks.push(block.text);
            }
        });

        html += `</div>`;
    });

    container.innerHTML = html;

    // --- TOP RIGHT COPY BUTTON ---
    const fullLearnText = learnBlocks.map(b => b.text).join(' ').trim();
    if (fullLearnText) {
        const copyBtn = document.createElement('div');
        copyBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="9" y="9" width="13" height="13" rx="2" ry="2"></rect><path d="M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1"></path></svg>`;
        copyBtn.title = "Copy To Clipboard";
        copyBtn.style.cssText = `
            position: absolute;
            top: 30px;
            right: 90px;
            color: white;
            opacity: 0.5;
            cursor: pointer;
            z-index: 200;
            padding: 10px;
            border-radius: 50%;
            background: rgba(255,255,255,0.1);
            transition: all 0.2s;
        `;

        copyBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            navigator.clipboard.writeText(fullLearnText).then(() => {
                const original = copyBtn.innerHTML;
                copyBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 24 24" fill="none" stroke="#4caf50" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="20 6 9 17 4 12"></polyline></svg>`;
                copyBtn.style.opacity = '1';
                setTimeout(() => {
                    copyBtn.innerHTML = original;
                    copyBtn.style.opacity = '0.5';
                }, 1500);
            });
        });

        copyBtn.addEventListener('mouseenter', () => { copyBtn.style.opacity = '1'; copyBtn.style.background = 'rgba(255,255,255,0.2)'; });
        copyBtn.addEventListener('mouseleave', () => { copyBtn.style.opacity = '0.5'; copyBtn.style.background = 'rgba(255,255,255,0.1)'; });

        overlay.appendChild(copyBtn);
    }

    // --- CLOSE BUTTON (X) ---
    const closeBtn = document.createElement('div');
    closeBtn.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>`;
    closeBtn.title = "Close (Space)";
    closeBtn.style.cssText = `
        position: absolute;
        top: 30px;
        right: 30px;
        color: white;
        opacity: 0.7;
        cursor: pointer;
        z-index: 200;
        padding: 8px;
        border-radius: 50%;
        background: rgba(255,255,255,0.1);
        transition: all 0.2s;
    `;
    closeBtn.addEventListener('click', (e) => {
        e.stopPropagation(); // Prevent overlay click if any
        togglePauseOverlay();
    });
    closeBtn.addEventListener('mouseenter', () => { closeBtn.style.opacity = '1'; closeBtn.style.background = 'rgba(255, 0, 0, 0.4)'; });
    closeBtn.addEventListener('mouseleave', () => { closeBtn.style.opacity = '0.7'; closeBtn.style.background = 'rgba(255,255,255,0.1)'; });
    overlay.appendChild(closeBtn);

    overlay.appendChild(container);

    // Add keydown listener for Space/Escape
    document.addEventListener('keydown', handleOverlayKeydown, true);

    // --- HELPER TO GET MOVIE METADATA ---


    // --- HELPER TO SAVE SUBTITLE ---
    const saveSubtitleFromState = async (annotations = null, overrideFullSentence = null, overrideTranslatedSentence = null) => {
        if (!activeCues || activeCues.length === 0) return;
        try {
            // Use pre-parsed blocks already collected in this scope (learnBlocks, knowBlocks, romaBlocks)

            const fullSentence = overrideFullSentence || learnBlocks.map(b => b.text).join(' ').trim();
            const translatedSentence = overrideTranslatedSentence || knowBlocks.join(' ').trim();
            const romanization = romaBlocks.join(' ').trim();

            if (fullSentence && currentLearnLang && annotations) {
                // --- METADATA EXTRACTION ---
                const { movieId, movieTitle, movieType, url } = await ensureMetadata();

                const payload = {
                    learn_language: currentLearnLang,
                    know_language: currentKnownLang,
                    sentence: fullSentence,
                    translated_sentence: translatedSentence,
                    romanization: romanization,
                    annotations: annotations, // Save the full annotation data
                    movie_title: movieTitle,
                    movie_url: url,
                    movie_type: movieType,
                    movie_id: movieId,
                    subtitle_start_ms: Math.floor(activeCues[0].start * 1000),
                    subtitle_end_ms: Math.floor(activeCues[activeCues.length - 1].end * 1000),
                };

                // console.debug("[LingoPause] Auto-saving subtitle (delayed)...", payload);

                chrome.runtime.sendMessage({
                    action: "SAVE_SUBTITLE",
                    data: payload
                }, (response) => {
                    if (response && response.success) {
                        const toast = document.createElement('div');
                        toast.innerText = "Saved to DB ✓";
                        toast.style.cssText = `
                                position: absolute;
                                top: 20px;
                                left: 50%;
                                transform: translateX(-50%);
                                background: rgba(40, 167, 69, 0.9);
                                color: white;
                                padding: 5px 15px;
                                border-radius: 20px;
                                font-size: 12px;
                                font-family: system-ui;
                                opacity: 0;
                                transition: opacity 0.5s;
                            `;
                        overlay.appendChild(toast);
                        toast.offsetHeight;
                        toast.style.opacity = '1';
                        setTimeout(() => toast.style.opacity = '0', 2000);
                    } else {
                        // console.warn("[LingoPause] Save failed:", response?.error);
                    }
                });

            }
        } catch (e) {
            console.error("[LingoPause] Auto-save error:", e);
        }
    };



    // Show spinner if we have learning components to annotate
    if (learnBlocks.length > 0 && currentLearnLang) {
        const spinner = document.createElement('div');
        spinner.id = 'multisubs-study-spinner';
        spinner.innerHTML = `
            <div class="multisubs-spinner-small"></div>
            <span>Loading annotations...</span>
        `;
        overlay.appendChild(spinner);

        // Cache object (memory) - ideally we'd use LRU but simple map is fine for session
        // We can also check chrome.storage.local for persistence if needed, but let's start with local var if it persists per page load
        // Actually, let's put it in a persistent scope or use storage

        const fetchAllAnnotations = async () => {
            let capturedAnnotations = null;
            try {
                // Combine all text with newlines
                const completeText = learnBlocks.map(b => b.text).join('\n');
                // console.debug("[LingoPause] Batch annotating text length:", completeText.length);

                // 1. Check Cache
                const cacheKey = `anno_${currentLearnLang}_${completeText.length}_${hashCode(completeText)}`;
                const cached = await getSubtitleFromStorage(cacheKey); // Re-use storage helper

                if (cached && Array.isArray(cached) && cached.length > 0) {
                    console.debug("[LingoPause] Using cached annotations for:", cacheKey);
                    await processAnnotations(cached);
                    // Also capture for save if needed (though we might not need to save if already saved?)
                    // The save logic expects capturedAnnotations to be populated
                    capturedAnnotations = cached;
                    return;
                }

                const formData = new FormData();
                formData.append('text', completeText);
                formData.append('language', currentLearnLang);
                formData.append('known_language', currentKnownLang);

                const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
                const token = supabase_session?.access_token;
                // console.debug("[LingoPause] Session found:", !!supabase_session, "Token present:", !!token);

                const headers = {};
                if (token) {
                    headers['Authorization'] = `Bearer ${token}`;
                }
                // console.debug("[LingoPause] Request headers:", headers);

                const response = await authenticatedFetch(`${CONFIG.BACKEND_BASE_URL}/annotate_sentence`, {
                    method: 'POST',
                    headers: headers,
                    body: formData
                });

                if (response.status === 429) {
                    try {
                        const errorData = await response.json();
                        if (errorData.detail === "Daily/Monthly sentence limit reached.") {
                            const alertDiv = document.createElement('div');
                            alertDiv.style.cssText = `
                                position: fixed;
                                top: 20px;
                                right: 20px;
                                background: #ff4444;
                                color: white;
                                padding: 15px;
                                border-radius: 8px;
                                z-index: 2147483647;
                                font-family: system-ui;
                                font-size: 16px;
                                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                            `;
                            alertDiv.innerHTML = `
                                <strong>[LingoPause] Pause Screen Quota reached</strong><br>
                                Your Pause Screen Quota for the month is used.<br>
                                Please upgrade plan.
                            `;
                            document.body.appendChild(alertDiv);
                            setTimeout(() => alertDiv.remove(), 2000);
                        }
                    } catch (e) {
                        console.error("[LingoPause] Error parsing 429 response:", e);
                    }
                }

                if (response.ok) {
                    const data = await response.json();
                    const allTokens = data.annotated_text; // Array of {token, meaning, ...}
                    capturedAnnotations = allTokens; // Capture for saving

                    // Save to cache
                    const cacheKey = `anno_${currentLearnLang}_${completeText.length}_${hashCode(completeText)}`;
                    saveSubtitleToStorage(cacheKey, allTokens);

                    await processAnnotations(allTokens);
                } else {
                    // console.error("[LingoPause] Batch annotation failed:", response.status);
                }
            } catch (e) {
                console.error("[LingoPause] Batch annotation error:", e);
            } finally {
                if (spinner.parentNode) {
                    spinner.remove();
                }
                // Call save AFTER annotations (finally block ensures it runs)
                saveSubtitleFromState(capturedAnnotations);
            }
        };

        // Helper to process and render annotations (shared by cache hit and fetch success)
        const processAnnotations = async (allTokens) => {
            let tokenIndex = 0;

            // Greedily assign tokens back to blocks
            for (const block of learnBlocks) {
                const blockTokens = [];
                let accumulatedText = "";

                const targetClean = block.text.replace(/\s/g, '');
                let currentClean = "";

                while (tokenIndex < allTokens.length && currentClean.length < targetClean.length) {
                    const node = allTokens[tokenIndex];
                    blockTokens.push(node);
                    currentClean += node.token.replace(/\s/g, '');
                    tokenIndex++;
                }

                // Render valid block tokens
                if (blockTokens.length > 0) {
                    const annotatedHtml = await Promise.all(blockTokens.map(async node => {
                        const pronunciation = await getRomanizationForToken(node.token, currentLearnLang, currentKnownLang);
                        node.romanization = pronunciation; // Capture romanization in the token object for saving
                        return `
                            <div style="display: inline-flex; flex-direction: column; align-items: center; margin: 0 4px; vertical-align: bottom;">
                                <div style="font-size: 3rem; color: #ffcc00; margin-bottom: 2px; max-width: 150px; line-height: 1.1; overflow-wrap: break-word; word-break: break-word; white-space: normal; text-align: center;">${node.meaning}</div>
                                <div style="font-size: 5rem; font-family: 'Noto Sans', 'Noto Sans CJK', sans-serif; line-height: 1.1;">${node.token}</div>
                                <div style="font-size: 3.2rem; color: #9fd3ff; font-family: 'Noto Sans', 'Noto Sans CJK', sans-serif; margin-top: 2px;">${pronunciation || ''}</div>
                            </div>
                        `;
                    }));

                    const target = document.getElementById(block.id);
                    if (target) {
                        target.innerHTML = `
                            <div style="display: flex; flex-wrap: wrap; justify-content: center; align-items: flex-end; margin-top: 8px;">
                                ${annotatedHtml.join('')}
                            </div>
                        `;
                    }
                }
            }
        };

        fetchAllAnnotations();
    } else {
        // No annotations to fetch, save immediately
        saveSubtitleFromState(null);
    }

    document.body.appendChild(overlay);
}

// Simple Hash Helper
function hashCode(str) {
    let hash = 0;
    if (str.length === 0) return hash;
    for (let i = 0; i < str.length; i++) {
        const chr = str.charCodeAt(i);
        hash = ((hash << 5) - hash) + chr;
        hash |= 0; // Convert to 32bit integer
    }
    return hash;
}


// function togglePauseOverlay() {
//     console.debug("togglePauseOverlay");
//     const overlayId = 'multisubs-overlay';
//     let overlay = document.getElementById(overlayId);
//     const video = document.querySelector('video');

//     if (overlay) {
//         // Overlay exists: remove it and play video
//         overlay.remove();
//         if (video) video.play();
//     } else {
//         // Overlay does not exist: pause video and create overlay
//         if (video) video.pause();

//         // Create overlay element
//         overlay = document.createElement('div');
//         overlay.id = overlayId;
//         overlay.style.cssText = `
//             position: fixed;
//             top: 0;
//             left: 0;
//             width: 100vw;
//             height: 100vh;
//             background-color: black;
//             z-index: 2147483647;
//             display: flex;
//             flex-direction: column;
//             justify-content: center;
//             align-items: center;
//             color: white;
//             font-family: Roboto, Arial, sans-serif;
//             text-align: center;
//             pointer-events: auto;
//             cursor: pointer;
//         `;

//         // Get active subtitles
//         let subtitleHTML = "";
//         if (video) {
//             const track = video.querySelector("track.multisubs-track");
//             if (track && track.track && track.track.activeCues && track.track.activeCues.length > 0) {
//                 // Combine text from all active cues
//                 for (let i = 0; i < track.track.activeCues.length; i++) {
//                     const cue = track.track.activeCues[i];
//                     let text = cue.text;
//                     // Simply replace newlines with breaks
//                     text = text.replace(/\n/g, '<br>');
//                     // Convert VTT class syntax <c.know-lang>...</c> to HTML span with style
//                     text = text.replace(/<c\.know-lang>(.*?)<\/c>/g, '<span style="color: #cccccc; font-size: 0.7em; display: block; margin-top: 10px;">$1</span>');
//                     // Convert VTT class syntax <c.pinyin>...</c>
//                     text = text.replace(/<c\.pinyin>(.*?)<\/c>/g, '<span style="color: #aaddff; font-size: 0.8em; display: block; margin-bottom: 5px;">$1</span>');

//                     subtitleHTML += (subtitleHTML ? "<br><br>" : "") + text;
//                 }
//             }
//         }

//         if (!subtitleHTML) {
//             subtitleHTML = "<div style='color: #666;'>No subtitles active</div>";
//         }

//         const textContainer = document.createElement('div');
//         textContainer.style.fontSize = '3rem';
//         textContainer.style.lineHeight = '1.2';
//         textContainer.style.padding = '20px';
//         textContainer.innerHTML = subtitleHTML;

//         overlay.appendChild(textContainer);

//         // Click to close
//         overlay.addEventListener('click', togglePauseOverlay);

//         document.body.appendChild(overlay);
//     }
// }
// Fixed TTML to VTT converter for Netflix
function isForcedNarrativeFromTTML(ttmlDoc) {
    return ttmlDoc
        ?.querySelector('metadata[nttm\\:textType]')
        ?.getAttribute('nttm:textType') === 'FN';
}

function convertTTMLToVTT(xmlString) {
    console.debug("[LingoPause] Starting TTML to VTT conversion (High Precision Mode)");
    try {
        const xml = new DOMParser().parseFromString(xmlString, 'text/xml');

        // Check for Forced Narrative (FN)
        if (isForcedNarrativeFromTTML(xml)) {
            console.debug("[LingoPause] Detected Forced Narrative (FN) in TTML. Discarding.");
            return "";
        }

        let vtt = "WEBVTT\n\n";

        const formatSecondsToVTT = (seconds) => {
            const h = Math.floor(seconds / 3600);
            const m = Math.floor((seconds % 3600) / 60);
            const s = Math.floor(seconds % 60);
            const ms = Math.floor((seconds % 1) * 1000);
            return `${String(h).padStart(2, '0')}:${String(m).padStart(2, '0')}:${String(s).padStart(2, '0')}.${String(ms).padStart(3, '0')}`;
        };

        const LINE_SELECTOR = 'tt > body > div > p, p';
        const paragraphs = xml.querySelectorAll(LINE_SELECTOR);
        console.debug("[LingoPause] Found paragraphs:", paragraphs.length);

        paragraphs.forEach((line) => {
            let text = '';
            const extractTextRecur = (parentNode) => {
                [].forEach.call(parentNode.childNodes, (node) => {
                    if (node.nodeType === 1) { // Node.ELEMENT_NODE
                        if (node.nodeName.toLowerCase() === 'br') text += '\n';
                        else extractTextRecur(node);
                    } else if (node.nodeType === 3) { // Node.TEXT_NODE
                        text += node.nodeValue + ' ';
                    }
                });
            };
            extractTextRecur(line);

            const beginAttr = line.getAttribute('begin');
            const endAttr = line.getAttribute('end');

            if (beginAttr && endAttr) {
                let beginSecs, endSecs;

                // Netflix uses tick format with 't' suffix
                // The tick rate needs to be determined from the header
                // Default Netflix tick rate is often 10000000 ticks per second
                const parseNetflixTime = (timeStr) => {
                    if (timeStr.endsWith('t')) {
                        // Remove 't' and parse as ticks
                        const ticks = parseInt(timeStr.slice(0, -1));

                        // Need to get tick rate from XML header
                        // Look for ttp:tickRate or ttp:frameRate
                        const ttElement = xml.querySelector('tt');
                        let tickRate = 10000000; // Default Netflix tick rate

                        if (ttElement) {
                            const tickRateAttr = ttElement.getAttribute('ttp:tickRate') ||
                                ttElement.getAttributeNS('http://www.w3.org/ns/ttml#parameter', 'tickRate');
                            if (tickRateAttr) {
                                tickRate = parseInt(tickRateAttr);
                            }
                        }

                        return ticks / tickRate;
                    } else if (!isNaN(timeStr)) {
                        // Pure number, assume ticks with default rate
                        return parseInt(timeStr) / 10000000;
                    } else {
                        // Try parsing as time string HH:MM:SS.mmm
                        const parts = timeStr.replace(',', '.').split(':');
                        if (parts.length === 3) {
                            const [h, m, s] = parts;
                            return parseInt(h) * 3600 + parseInt(m) * 60 + parseFloat(s);
                        }
                        return 0;
                    }
                };

                beginSecs = parseNetflixTime(beginAttr);
                endSecs = parseNetflixTime(endAttr);

                vtt += `${formatSecondsToVTT(beginSecs)} --> ${formatSecondsToVTT(endSecs)}\n${text.trim()}\n\n`;
            }
        });

        console.debug("[LingoPause] TTML conversion complete.");
        console.debug("vtt", vtt);
        return vtt;
    } catch (e) {
        console.error("[LingoPause] Error converting TTML to VTT:", e);
        return "";
    }
}

// --- REUSABLE AUTH FETCH HELPER ---
const authenticatedFetch = async (url, options = {}) => {
    let headers = options.headers || {};

    // Initial Request
    // console.debug(`[LingoPause] Fetching ${url}...`);
    let response = await fetch(url, options);
    // console.debug(`[LingoPause] Initial response status: ${response.status}`);

    // If 401 Unauthorized, try to refresh session
    if (response.status === 401) {
        console.warn("[LingoPause] 401 Unauthorized. body:", await response.clone().text());
        console.warn("[LingoPause] Attempting session refresh...");

        const refreshResponse = await new Promise(resolve =>
            chrome.runtime.sendMessage({ action: "REFRESH_SESSION" }, resolve)
        );

        // console.debug("[LingoPause] Refresh response:", refreshResponse);

        if (refreshResponse && refreshResponse.success && refreshResponse.session) {
            // console.debug("[LingoPause] Session refreshed. Retrying request...");
            const newToken = refreshResponse.session.access_token;

            // Update Authorization header
            headers = {
                ...headers,
                'Authorization': `Bearer ${newToken}`
            };

            // Clone options and update headers
            const newOptions = { ...options, headers: headers };

            // Retry Request
            response = await fetch(url, newOptions);
            // console.debug(`[LingoPause] Retry response status: ${response.status}`);
        } else {
            console.error("[LingoPause] Refresh failed during retry. Reason:", refreshResponse?.error);

            // Show session expired toast
            if (!document.getElementById('multisubs-session-toast')) {
                const toast = document.createElement('div');
                toast.id = 'multisubs-session-toast';
                toast.innerText = "Session expired. Please sign in via the extension.";
                toast.style.cssText = `
                    position: fixed;
                    top: 20px;
                    left: 50%;
                    transform: translateX(-50%);
                    background: rgba(220, 53, 69, 0.95);
                    color: white;
                    padding: 8px 16px;
                    border-radius: 4px;
                    font-size: 14px;
                    font-family: system-ui, -apple-system, sans-serif;
                    z-index: 2147483647;
                    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
                    pointer-events: none;
                `;
                document.body.appendChild(toast);
                setTimeout(() => {
                    toast.style.transition = 'opacity 0.5s';
                    toast.style.opacity = '0';
                    setTimeout(() => toast.remove(), 500);
                }, 4000);
            }
        }
    }
    return response;
};

// --- USAGE TRACKING ---
class UsageTracker {
    constructor() {
        this.accumulatedSeconds = 0;
        this.lastTickTime = null;
        this.trackingInterval = null;
        this.SYNC_INTERVAL_MS = CONFIG.SYNC_INTERVAL_MS; // 60 seconds
        this.syncTimer = null;
        this.isDisabled = false;
        this.videoElement = null;
        this.hasShownLimitError = false;
        this.isInitialized = false;

        // Bind methods once for consistent references (allows removal)
        this.boundHandlePlay = this.handlePlay.bind(this);
        this.boundHandlePause = this.handlePause.bind(this);
        this.boundHandleTick = this.handleTick.bind(this);
    }

    refreshVideo() {
        if (this.isDisabled) return;
        const newVideo = document.querySelector('video');

        // If we found a NEW video, or if we had no video before
        if (newVideo && newVideo !== this.videoElement) {
            console.debug(`[LingoPause] UsageTracker: Video changed. Found: ${newVideo.tagName}. Paused: ${newVideo.paused}`);

            // Detach from old if exists
            if (this.videoElement) {
                this.videoElement.removeEventListener('play', this.boundHandlePlay);
                this.videoElement.removeEventListener('pause', this.boundHandlePause);
                this.videoElement.removeEventListener('ratechange', this.boundHandleTick);
            }

            // Attach to new
            this.videoElement = newVideo;
            this.videoElement.addEventListener('play', this.boundHandlePlay);
            this.videoElement.addEventListener('pause', this.boundHandlePause);
            this.videoElement.addEventListener('ratechange', this.boundHandleTick);

            // If the new video is ALREADY playing, we need to catch up
            // if (!this.videoElement.paused) {
            //     console.debug("[LingoPause] UsageTracker: Video already playing, forcing handlePlay.");
            //     this.handlePlay();
            // }
        } else if (!newVideo) {
            console.debug("[LingoPause] UsageTracker: No video found in refreshVideo.");
        }
    }

    init() {
        if (this.isDisabled) return;
        this.refreshVideo(); // Try to find video and attach
        if (this.videoElement && !this.videoElement.paused) { // only for youtube
            console.debug("[LingoPause] UsageTracker: Video already playing, forcing handlePlay.");
            this.handlePlay();
        }
        // Start periodic sync if not running
        if (!this.syncTimer) {
            this.syncTimer = setInterval(this.syncUsage.bind(this), this.SYNC_INTERVAL_MS);
        }

        // Global unload listener (only attach once)
        if (!this.isInitialized) {
            window.addEventListener('beforeunload', () => {
                this.handlePause();
                this.syncUsage(true);
            });
            this.isInitialized = true;
            console.debug("[LingoPause] UsageTracker initialized (20s sync)");
        }
    }

    enable() {
        if (!this.isDisabled && this.isInitialized && this.videoElement === document.querySelector('video')) return;
        console.debug("[LingoPause] Re-enabling UsageTracker...");
        this.isDisabled = false;
        this.hasShownLimitError = false;
        this.trackingInterval = false;
        this.init(); // init will call refreshVideo to attach if needed
    }

    handlePlay() {
        if (this.isDisabled) return;
        // If we are already tracking, don't restart the interval or reset the tick time!
        // This causes fractional seconds to be lost if handlePlay is called repeatedly.
        if (this.trackingInterval) {
            console.debug("[LingoPause] UsageTracker: handlePlay called but already tracking. Ignoring.");
            return;
        }

        console.debug("[LingoPause] UsageTracker: handlePlay (Interval starting)");
        this.lastTickTime = Date.now();
        // Use a 1-second interval to accumulate time while playing
        // This is more robust than just play/pause timestamps if the user seeks or networking lags
        if (this.trackingInterval) clearInterval(this.trackingInterval);
        this.trackingInterval = setInterval(this.boundHandleTick, 1000);
    }

    handlePause() {
        if (this.isDisabled) return;
        console.debug("[LingoPause] UsageTracker: handlePause");
        this.handleTick(); // Add remaining time
        if (this.trackingInterval) clearInterval(this.trackingInterval);
        this.trackingInterval = null;
        this.lastTickTime = null;
        this.syncUsage(); // Sync on pause
    }

    handleTick() {
        if (this.isDisabled || !this.lastTickTime) return;
        const now = Date.now();
        // Calculate delta with 1 decimal place precision (e.g. 1.1s)
        const deltaSeconds = Math.round((now - this.lastTickTime) / 100) / 10;

        // Sanity check: cap delta at 5 seconds (in case of weird sleep/resume behavior)
        if (deltaSeconds > 0 && deltaSeconds < 5) {
            this.accumulatedSeconds += deltaSeconds;
            // Round accumulated total to 1 decimal place to avoid float drift like 1.00000012
            this.accumulatedSeconds = Math.round(this.accumulatedSeconds * 10) / 10;
            // console.debug(`[LingoPause] Tick: +${deltaSeconds}s -> ${this.accumulatedSeconds}s`);
        } else if (deltaSeconds >= 5) {
            console.warn(`[LingoPause] Usage delta rejected: ${deltaSeconds.toFixed(1)}s`);
        }
        this.lastTickTime = now;
    }

    async syncUsage(isBeacon = false) {
        try {
            const { isActive } = await chrome.storage.sync.get(['isActive']);
            // DEBUG LOG
            console.debug(`[LingoPause] syncUsage called. isActive: ${isActive}, isDisabled: ${this.isDisabled}, accumulated: ${this.accumulatedSeconds}`);

            if (!this.isDisabled && (!hasActiveLearnSubtitles || !hasActiveKnowSubtitles)) {
                // If we don't have BOTH subtitles successfully loaded, we do not charge usage.
                // Also reset accumulated time so we don't accidentally charge for time spent waiting for a fix.
                this.accumulatedSeconds = 0;
                return;
            } else {
                console.log("[LingoPause] sync Usage: has Learn and Know subtitles");
            }

            // If disabled, we only retry (ping with 0s). If enabled, we need accumulated usage.
            if (!isActive) return;

            // Only sync if we have at least 5 seconds, OR if it's a beacon (forced sync on unload), OR if we are checking limit (disabled)
            if (!this.isDisabled && !isBeacon && this.accumulatedSeconds < 5) {
                // console.debug("[LingoPause] Accumulated usage < 5s, skipping sync.");
                return;
            }

            const secondsToSend = this.isDisabled ? 0 : Math.floor(this.accumulatedSeconds);
            // Deduct what we're about to send immediately to avoid double counting if async lags
            if (!this.isDisabled) {
                this.accumulatedSeconds -= secondsToSend;
                // Round again after subtraction
                this.accumulatedSeconds = Math.round(this.accumulatedSeconds * 10) / 10;
            }

            // try { <--- Removing this line effectively (or commenting it out to be safe, but removal is cleaner)
            // I will replace it with empty string to remove it, but I need to match the indentation.

            // console.debug(`[LingoPause] Syncing ${secondsToSend}s...`);
            const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
            const token = supabase_session?.access_token;

            if (!token) {
                console.warn("[LingoPause] Usage tracking skipped: No auth token");
                return;
            }

            const payload = JSON.stringify({ seconds: secondsToSend });
            const url = `${CONFIG.BACKEND_BASE_URL}/track_dual_subs_usage`;

            if (isBeacon) {
                // Use Navigator.sendBeacon if possible for page unload, but it doesn't support custom auth headers easily.
                // Fallback to fetch with keepalive.
                fetch(url, {
                    method: 'POST',
                    headers: {
                        'Authorization': `Bearer ${token}`,
                        'Content-Type': 'application/json'
                    },
                    body: payload,
                    keepalive: true
                }).catch(e => console.error("[LingoPause] Beacon failed", e));
            } else {
                const response = await authenticatedFetch(url, {
                    method: 'POST',
                    headers: {
                        'Authorization': `Bearer ${token}`,
                        'Content-Type': 'application/json'
                    },
                    body: payload
                });

                if (response.status === 429) {
                    this.disableDualSubtitles('usage_limit', supabase_session?.user?.id);
                    return;
                }

                // If still 401 after retry, it means refresh failed or token invalid
                if (response.status === 401) {
                    console.warn("[LingoPause] Refresh failed/Auth invalid, stopping tracker.");
                    this.disableDualSubtitles('auth_invalid');
                    return;
                }

                if (!response.ok) {
                    console.warn(`[LingoPause] Usage validation failed: ${response.status}`);
                    if (!this.isDisabled) this.accumulatedSeconds += secondsToSend; // Restore usage on generic error
                    this.disableDualSubtitles('error');
                    return;
                } else {
                    // Success!
                    if (this.isDisabled) {
                        console.debug("[LingoPause] Connection/Limit restored. Re-enabling...");
                        // Use enable() to ensure listeners/intervals are restarted properly!
                        this.enable();

                        // Clear the local storage limit flag for this user immediately
                        const currentUserId = supabase_session?.user?.id;
                        if (currentUserId) {
                            chrome.storage.local.get(['userUsageLimits'], (result) => {
                                const limits = result.userUsageLimits || {};
                                if (limits[currentUserId]) {
                                    delete limits[currentUserId];
                                    chrome.storage.local.set({ userUsageLimits: limits });
                                }
                            });
                        }

                        // Re-trigger subtitle check to restore UI
                        if (typeof checkConditionsForDualSubtitles === 'function') {
                            checkConditionsForDualSubtitles();
                        } else {
                            // Fallback if function not in scope (it usually isn't in content-helper, so we might need a reload or a message)
                            // Actually better to just reload the page or let the user click play?
                            // Let's reload to be safe and clean or send message to content.js?
                            // For now, let's just clear the flag. The user might need to refresh or toggle.
                            // Actually, we can try to re-run ensuresubtitlecontainer loop?
                            // The renderer loop was stopped. We need to restart it.
                            // Implementing a simple "auto-resume" might be complex.
                            // Showing a toast "Limit restored, please refresh" might be easier?
                            // OR, we just let the next play() event handle it?
                            // The variable 'isDisabled' is false now. Next play() will start tracking.
                            // But the UI was removed.
                        }
                    }

                    chrome.storage.local.remove(['usageLimitReached', 'usageLimitUserId']); // Clean up legacy
                    console.debug(`[LingoPause] Synced ${secondsToSend}s of usage.`);
                }
            }
        } catch (e) {
            if (e.message.includes("Extension context invalidated")) {
                console.warn("[LingoPause] Extension context invalidated (extension reloaded?). Stopping tracker.");
                if (typeof showErrorIndicator === 'function') {
                    showErrorIndicator("[LingoPause] Extension updated. Please refresh the page to continue.", this.videoElement);
                }
                this.isDisabled = true;
                if (this.trackingInterval) clearInterval(this.trackingInterval);
                if (this.syncTimer) clearInterval(this.syncTimer);
            } else {
                console.error("[LingoPause] Sync usage error:", e);
                // On error, we still want to disable to be safe
                this.disableDualSubtitles('error');
            }
        }
    }


    async checkAllowance() {
        const { isActive } = await chrome.storage.sync.get(['isActive']);
        if (!isActive) return false;

        try {
            const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
            const token = supabase_session?.access_token;
            if (!token) return false;

            // Send 0 seconds to check limit without incrementing
            const payload = JSON.stringify({ seconds: 0 });
            const url = `${CONFIG.BACKEND_BASE_URL}/track_dual_subs_usage`;

            const response = await authenticatedFetch(url, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                },
                body: payload
            });

            if (response.status === 429) {
                this.disableDualSubtitles('usage_limit', supabase_session?.user?.id);
                return false;
            }
            if (!response.ok) {
                this.disableDualSubtitles('error');
                return false;
            }

            // Success! Heal local state if needed
            const currentUserId = supabase_session?.user?.id;
            if (currentUserId) {
                chrome.storage.local.get(['userUsageLimits'], (result) => {
                    const limits = result.userUsageLimits || {};
                    if (limits[currentUserId]) {
                        console.debug("[LingoPause] checkAllowance passed. Clearing stale local limit flag.");
                        delete limits[currentUserId];
                        chrome.storage.local.set({ userUsageLimits: limits });
                    }
                });
            }

            return true;
        } catch (e) {
            console.error("[LingoPause] checkAllowance error:", e);
            this.disableDualSubtitles('error');
            return false;
        }
    }

    disableDualSubtitles(reason = 'usage_limit', userId = null) {
        this.isDisabled = true;
        if (this.trackingInterval) clearInterval(this.trackingInterval);

        // Stop periodic usage syncing/checking (as requested: only check on reload/new video)
        // Previous logic only cleared for auth_invalid, now clear for EVERYTHING to stop polling.
        // if (this.syncTimer) {
        //     console.debug("[LingoPause] Clearing syncTimer (polling stopped).");
        //     clearInterval(this.syncTimer);
        //     this.syncTimer = null;
        // }

        console.warn("[LingoPause] Usage limit reached. Disabling features.");

        // Remove subtitles UI
        removeDualSubtitles(); // NEW: Clear HTML overlay
        nukeAllSubtitles(); // Legacy clear

        // Notify user if not already shown (for limits)
        if (reason === 'usage_limit' || reason === 'error') {
            if (this.hasShownLimitError) return;
            this.hasShownLimitError = true;
        }

        if (reason === 'usage_limit' && userId) {
            chrome.storage.local.get(['userUsageLimits'], (result) => {
                const limits = result.userUsageLimits || {};
                limits[userId] = true;
                chrome.storage.local.set({ userUsageLimits: limits });
            });
            showErrorIndicator("Dual subtitles limit reached.", this.videoElement);
        } else if (reason === 'error') {
            showErrorIndicator("Error connecting to LingoPause.", this.videoElement);
        } else if (reason === 'auth_invalid') {
            showErrorIndicator("Session Expired. Please log in.", this.videoElement);
        }
    }
}

// Instantiate and start tracking
const usageTracker = new UsageTracker();
setTimeout(() => {
    usageTracker.init();
}, 5000);

