(function () {
    let currentUrl = location.href;

    const verifyLimitWithServer = async () => {
        const { supabase_session, userUsageLimits } = await chrome.storage.local.get(['supabase_session', 'userUsageLimits']);
        const token = supabase_session?.access_token;
        const currentUserId = supabase_session?.user?.id;

        if (!token || !currentUserId) return;

        // Only verify if we think there IS a limit
        const isLimitApplicable = userUsageLimits && userUsageLimits[currentUserId];
        if (!isLimitApplicable) return;

        console.debug("[LingoPause] Verifying limit validity with server...");
        try {
            // Need CONFIG base URL. Assuming available gloabally or hardcoded fallback?
            // content-helper.js defines CONFIG. We are in content.js.
            // content.js might not have access to CONFIG if it's not injected in same context scope easily.
            // But content-helper.js is injected alongside.
            // Wait, CONFIG is in config.js. 
            // Let's use chrome.runtime.getURL if needed or assume CONFIG exists (it does in popup context, but here?).
            // Actually content.js runs isolated. We might need to fetch from storage if stored, or hardcode/import.
            // Safer to assume CONFIG is injected. If not, use localhost default or try to get from somewhere.
            // Re-using the logic from content-helper which seems to have it.
            // Wait, checks above use CONFIG.BACKEND_BASE_URL... No wait, usageTracker uses it in content-helper.
            // content.js does NOT seemingly use CONFIG elsewhere. 
            // Ah, displayDualSubtitles uses send_ytplayer / fetch_and_merge_subtitles which are in content-helper.
            // Let's rely on content-helper or just hardcode/inject properly.
            // Actually, we can just use usageTracker.checkAllowance() if usageTracker was exposed? 
            // It is not exposed to content.js scope directly (isolated worlds vs standard?). 
            // They are both content scripts. content-helper runs in the page? No, manifest says "js": ["config.js", "content-helper.js", "content.js"].
            // So they share range.

            // Re-implement simplified fetch here to depend less on complex classes
            const BASE_URL = (typeof CONFIG !== 'undefined') ? CONFIG.BACKEND_BASE_URL : 'http://localhost:3000'; // Fallback

            const response = await fetch(`${BASE_URL}/track_dual_subs_usage`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ seconds: 0 })
            });

            if (response.ok) {
                console.log("[LingoPause] Server verified limit is lifted! Clearing local flag.");
                if (userUsageLimits[currentUserId]) {
                    delete userUsageLimits[currentUserId];
                    await chrome.storage.local.set({ userUsageLimits });
                    // Re-run check conditions to auto-start
                    // checkConditionsForDualSubtitles();
                }
            } else {
                console.log("[LingoPause] Server confirmed limit is still active (or error). Status:", response.status);
            }
        } catch (e) {
            console.error("[LingoPause] verifyLimitWithServer failed:", e);
        }
    };

    const checkConditionsForDualSubtitles = () => {
        console.debug("[LingoPause] Checking conditions for dual subtitles...");
        try {
            chrome.storage.sync.get(['isActive', 'learnLang', 'knowLang'], (result) => {
                if (chrome.runtime.lastError) {
                    console.warn("[LingoPause] Storage access error:", chrome.runtime.lastError);
                    if (chrome.runtime.lastError.message && chrome.runtime.lastError.message.includes("Extension context invalidated") && typeof showErrorIndicator === 'function') {
                        showErrorIndicator("[LingoPause] Extension updated. Please refresh the page to continue.");
                    }
                    return;
                }
                const { isActive, learnLang, knowLang } = result;

                // Also check local storage for usage limit and session
                chrome.storage.local.get(['userUsageLimits', 'supabase_session'], (localResult) => {
                    const userUsageLimits = localResult.userUsageLimits || {};
                    const currentUserId = localResult.supabase_session?.user?.id;

                    // Check if current user is in the limits map
                    const isLimitApplicable = currentUserId && userUsageLimits[currentUserId];

                    const isYoutube = location.hostname.includes('youtube.com');
                    const isNetflix = location.hostname.includes('netflix.com');
                    const isVideoPage = (isYoutube && location.pathname === '/watch') || (isNetflix && location.pathname.startsWith('/watch/'));

                    if (isActive && isVideoPage && learnLang && knowLang && !isLimitApplicable) {
                        displayDualSubtitles(learnLang, knowLang, isYoutube ? 'youtube' : 'netflix');
                    } else {
                        if (isLimitApplicable) {
                            console.warn("[LingoPause] Usage limit reached (local cache). verifying...");
                            // We are blocked locally. BUT we should check server if this was a refresh/init.
                            // However, we don't want to loop. 
                            // VerifyLimitWithServer calls checkConditions on success.
                            // So here we simply stop. The initial load or toggle will have triggered verifyLimitWithServer separately.
                        }
                        removeDualSubtitles();
                    }
                });
            });
        } catch (e) {
            console.warn("[LingoPause] checkConditionsForDualSubtitles error:", e);
            if (e.message && e.message.includes("Extension context invalidated") && typeof showErrorIndicator === 'function') {
                showErrorIndicator("[LingoPause] Extension updated. Please refresh the page to continue.");
            }
        }
    };



    const displayDualSubtitles = async (learnLang, knowLang, platform) => {
        console.log("[LingoPause] display dual subtitles");
        const video = document.querySelector('video');
        if (video && video.paused && !video.ended && video.readyState > 1) {
            if (video.dataset.lpWaitingForPlay === 'true') {
                console.debug("[LingoPause] Already waiting for play.");
                return;
            }
            console.debug("[LingoPause] Video is paused. Waiting for play event to fetch subtitles...");
            video.dataset.lpWaitingForPlay = 'true';

            const onPlay = () => {
                console.debug("[LingoPause] Play detected. Resuming...");
                delete video.dataset.lpWaitingForPlay;
                checkConditionsForDualSubtitles();
            };
            video.addEventListener('play', onPlay, { once: true });
            return;
        }

        const { supabase_session } = await chrome.storage.local.get(['supabase_session']);
        if (!supabase_session?.access_token) {
            console.debug("[LingoPause] User not logged in, skipping dual subtitles.");
            removeDualSubtitles();
            return;
        }

        removeDualSubtitles();
        // await showLoadingIndicator(); // Moved to after video check
        // console.debug(`[LingoPause] loading dual subtitles for ${platform}`);
        try {
            if (platform === 'youtube') {
                await showLoadingIndicator();
                const plyRes = await send_ytplayer();
                if (plyRes) {
                    // Wait for ad to finish
                    const AD_WAIT_TIMEOUT = 60000;
                    const adWaitStart = Date.now();
                    while (document.querySelector("div.ad-showing") && Date.now() - adWaitStart < AD_WAIT_TIMEOUT) {
                        console.debug("[LingoPause] Waiting for ad to finish...");
                        await sleep(500);
                    }

                    // await sleep(5000);
                    // console.debug("[LingoPause] sleep for 5s");
                    await fetch_and_merge_subtitles(learnLang, knowLang);
                }
            } else if (platform === 'netflix') {
                await showLoadingIndicator();
                // For Netflix, cachedCaptionTracks should be populated by the interceptor
                await fetch_and_merge_subtitles(learnLang, knowLang);
            }
        } catch (error) {
            console.error("Error in displayDualSubtitles:", error);

            if (error?.message && error?.message.includes("Extension context invalidated") && typeof showErrorIndicator === 'function') {
                showErrorIndicator("[LingoPause] Extension updated. Please refresh the page to continue.");
                return;
            }


            showErrorIndicator(
                "Unable to retrieve subtitles: " + (error?.message || error)
            );


        }
        finally {
            hideLoadingIndicator();
        }
    }



    // Inject Netflix Interceptor if on Netflix
    if (location.hostname.includes('netflix.com')) {
        //  console.debug("[LingoPause] Netflix detected, preparing interceptor...");

        let lastMovieId = null;

        // Listen BEFORE injecting
        window.addEventListener('netflix-manifest-captured', (e) => {
            // console.debug("[LingoPause] Received netflix-manifest-captured event!", e.detail.tracks?.length, "tracks found");

            const currentMovieId = e.detail.movieId;
            if (currentMovieId && currentMovieId !== lastMovieId) {
                // New movie, clear cache
                console.debug("[LingoPause] New movie detected, clearing caption cache. Old:", lastMovieId, "New:", currentMovieId);
                cachedCaptionTracks = [];
                if (typeof clearSubtitleContentCache === 'function') {
                    clearSubtitleContentCache();
                }
                lastMovieId = currentMovieId;
            } else {
                // Same movie, we will merge. Do not clear.
                // console.debug("[LingoPause] Same movie manifest update. Merging tracks.");
            }
            if (typeof setupNetflixSubtitleData === 'function') {
                setupNetflixSubtitleData(e.detail.tracks);
                // Verify limit when new netflix data arrives (video load)
                verifyLimitWithServer();
                checkConditionsForDualSubtitles();
            } else {
                console.error("[LingoPause] ERROR: setupNetflixSubtitleData is NOT defined!");
            }
        });

        const script = document.createElement('script');
        script.src = chrome.runtime.getURL('netflix-interceptor.js');
        (document.head || document.documentElement).appendChild(script);
        script.onload = () => {
            // console.debug("[LingoPause] netflix-interceptor.js injected and loaded");
            script.remove();
        };
    }

    // Initial check
    if (location.hostname.includes('youtube.com')) {
        verifyLimitWithServer().then(() => {
            checkConditionsForDualSubtitles();
        });
    }

    // Monitor URL changes (YouTube is a SPA, so we need to observe navigation)
    const observer = new MutationObserver(() => {
        if (location.href !== currentUrl) {
            // console.debug("[LingoPause] ========================================");
            // console.debug("[LingoPause] URL CHANGE DETECTED");
            // console.debug("[LingoPause] Old URL:", currentUrl);
            // console.debug("[LingoPause] New URL:", location.href);
            // console.debug("[LingoPause] ========================================");

            currentUrl = location.href;

            // On Netflix, navigating away from a video means we need to clear the MANIFEST cache
            // if (location.hostname.includes('netflix.com')) {
            //     console.debug("[LingoPause] Netflix Navigation: Clearing caption cache for new video");
            //     if (typeof cachedCaptionTracks !== 'undefined') {
            //         cachedCaptionTracks = [];
            //     }
            // }

            if (location.hostname.includes('netflix.com')) {
                removeDualSubtitles();



                setTimeout(() => {
                    // Re-inject interceptor for new video
                    // console.debug("Re-inject interceptor for new video");
                    const script = document.createElement('script');
                    script.src = chrome.runtime.getURL('netflix-interceptor.js');
                    (document.head || document.documentElement).appendChild(script);
                    script.onload = () => script.remove();
                }, 1500);
            }



            removeDualSubtitles();
            setTimeout(() => {
                // Re-verify limit on video change just in case
                verifyLimitWithServer();
                // Ensure tracker is attached to new video element
                if (typeof usageTracker !== 'undefined') {
                    usageTracker.refreshVideo();
                }
                checkConditionsForDualSubtitles();
            }, 1000); // Wait a bit longer for Netflix UI to settle
        }
    });

    observer.observe(document, { subtree: true, childList: true });

    // Listen for Ctrl+Q to toggle pause overlay
    document.addEventListener('keydown', (e) => {
        if ((e.ctrlKey || e.metaKey) && (e.key === 'q' || e.key === 'Q')) {
            togglePauseOverlay();
        }
    });

    // Monitor storage changes (if user toggles extension on/off while on the page)
    chrome.storage.onChanged.addListener((changes, namespace) => {
        if (namespace === 'sync') {
            // Special handling for isActive toggle ON
            if (changes.isActive && changes.isActive.newValue === true) {
                console.log("[LingoPause] Extension activated. Verifying usage limits...");
                // This calls verifyLimitWithServer. We also fall through to call checkConditionsForDualSubtitles
                // so that we don't block start-up if there are no limits or if verification is async.
                verifyLimitWithServer();
                // return; // Removed to allow fall-through so checkConditionsForDualSubtitles is always called
            }

            // For other changes (lang change, or toggle OFF), assume standard check
            const coreSettingsChanged = changes.isActive || changes.learnLang || changes.knowLang;
            if (coreSettingsChanged) {
                console.debug("[LingoPause] Core settings changed (not activation), re-evaluating...");
                checkConditionsForDualSubtitles();
            }
        } else if (namespace === 'local') {
            // Check for session changes or usage limit updates (from other tabs/popup)
            if (changes.supabase_session || changes.userUsageLimits) {
                console.debug("[LingoPause] Session/Limit changed, re-evaluating dual subtitles...");
                checkConditionsForDualSubtitles();
            }
        }
    });

})();
