(function () {
    // console.debug("[LingoPause] Netflix Interceptor Loaded");

    class DehydratedSubtitle {
        constructor(lang, bcp47) {
            this.lang = lang;
            this.bcp47 = bcp47;
            this.isHydrated = false;
        }
    }

    class ImageSubtitle {
        constructor(lang, bcp47, urls, isCaption) {
            this.lang = lang;
            this.bcp47 = bcp47;
            this.urls = urls;
            this.isCaption = isCaption;
            this.isImage = true;
        }
    }

    class TextSubtitle {
        constructor(lang, bcp47, urls, isCaption) {
            this.lang = lang;
            this.bcp47 = bcp47;
            this.urls = urls;
            this.isCaption = isCaption;
            this.isText = true;
        }
    }

    class SubtitleFactory {
        // track: manifest.textTracks[...]
        static build(track) {
            const isImageBased = Object.values(track.ttDownloadables).some(d => d.isImage);
            const isCaption = track.rawTrackType === 'closedcaptions';
            const lang = track.languageDescription + (isCaption ? ' [CC]' : '');
            const bcp47 = track.language;

            if (!track.hydrated) {
                return new DehydratedSubtitle(lang, bcp47);
            }
            // Prioritize TEXT over IMAGE
            // Some tracks might have both (rare), or we just want to ensure we use text if available
            const textSub = this._buildTextBased(track, lang, bcp47, isCaption);
            if (textSub) {
                return textSub;
            }

            if (isImageBased) {
                console.debug("[LingoPause] Image-based subtitle detected (and no text version found):", lang);
                return this._buildImageBased(track, lang, bcp47, isCaption);
            }

            return null;
        }

        static isNoneTrack(track) {
            // Filter forced narratives
            if (track.isForcedNarrative) {
                // console.debug("[LingoPause] Filtered forced narrative track: " + track.languageDescription);
                return true;
            }

            // Sometimes Netflix places "fake" text tracks into manifests.
            // Such tracks have "isNoneTrack: false" and even have downloadable URLs,
            // while their display name is "Off" (localized in UI language, e.g., "關閉").
            // Here we use a huristic rule concluded by observation to filter those "fake" tracks out.
            if (track.isNoneTrack) {
                return true;
            }

            // "new_track_id" example "T:1:0;1;zh-Hant;1;1;"
            // the last bit is 1 for NoneTrack text tracks
            try {
                const isNoneTrackBit = track.new_track_id.split(';')[4];
                if (isNoneTrackBit === '1') {
                    return true;
                }
            }
            catch (err) {
            }

            // "rank" === -1
            if (track.rank !== undefined && track.rank < 0) {
                return true;
            }
            return false;
        }

        static _buildImageBased(track, lang, bcp47, isCaption) {
            const maxHeight = Math.max(...Object.values(track.ttDownloadables).map(d => {
                if (d.height)
                    return d.height;
                else
                    return -1;
            }));
            const d = Object.values(track.ttDownloadables).find(d => d.height === maxHeight);
            let urls;
            if (d.downloadUrls) {
                urls = Object.values(d.downloadUrls);
            } else {
                urls = d.urls.map(t => t.url);
            }
            return new ImageSubtitle(lang, bcp47, urls, isCaption);
        }

        static _buildTextBased(track, lang, bcp47, isCaption) {
            // Try multiple profiles in order of preference
            const targetProfiles = ['dfxp-ls-sdh', 'imsc1.1']; //simplesdh, or add height is not a property
            let d = null;
            let foundProfile = null;

            for (const profile of targetProfiles) {
                if (track.ttDownloadables[profile]) {
                    d = track.ttDownloadables[profile];
                    foundProfile = profile;
                    break;
                }
            }

            if (!d) {
                console.debug(`Cannot find any of [${targetProfiles.join(', ')}] for ${lang}`);
                return null;
            }

            // console.debug(`WordWatch> Using profile "${foundProfile}" for ${lang}`);
            let urls;
            if (d.downloadUrls) {
                urls = Object.values(d.downloadUrls);
            } else {
                urls = d.urls.map(t => t.url);
            }
            return new TextSubtitle(lang, bcp47, urls, isCaption);
        }
    }

    class ManifestManagerInMemory {
        constructor() {
            this.manifests = {};
        }
        saveManifest(manifest) {
            if (manifest && manifest.movieId) this.manifests[manifest.movieId] = manifest;
        }
        getManifest(movieId) {
            return this.manifests[movieId];
        }
    }

    class WordWatchManager {
        constructor() {
            this.manifestManager = new ManifestManagerInMemory();
            this.lastMovieId = null;
        }

        updateManifest(manifest) {
            this.manifestManager.saveManifest(manifest);
            this.activateManifest(manifest.movieId);
        }

        activateManifest(movieId) {
            const manifest = this.manifestManager.getManifest(movieId);
            if (!manifest) return;

            // Dispatch event for content script
            if (manifest.timedtexttracks) {
                const tracks = manifest.timedtexttracks;
                const processedTracks = tracks
                    .filter(t => !SubtitleFactory.isNoneTrack(t))
                    .map(t => SubtitleFactory.build(t))
                    .filter(t => t !== null);

                window.dispatchEvent(new CustomEvent("netflix-manifest-captured", {
                    detail: {
                        tracks: processedTracks,
                        movieId: movieId
                    }
                }));
                // console.debug(`[LingoPause] Manifest dispatched for movieId: ${movieId}, tracks: ${processedTracks.length}`);
            }

            this.lastMovieId = movieId;
        }
    }

    window.__WordWatch = new WordWatchManager();

    // --- Hook JSON.parse to intercept manifests ---
    const _parse = JSON.parse;
    window.JSON.parse = function (...args) {
        const result = _parse.apply(this, args);
        try {
            if (result && typeof result === 'object') {
                // Check if it looks like a manifest response, even if structure is slightly different
                if (result.result && result.result.movieId) {
                    // console.debug("[LingoPause] JSON.parse intercepted manifest with movieId:", result.result.movieId);
                    window.__WordWatch.updateManifest(result.result);
                } else if (result.movieId && result.timedtexttracks) {
                    // Alternative structure?
                    console.debug("[LingoPause] JSON.parse intercepted ALTERNATIVE manifest structure with movieId:", result.movieId);
                    window.__WordWatch.updateManifest(result);
                }
            }
        } catch (err) {
            console.error("[LingoPause] Error in JSON.parse hook:", err);
        }
        return result;
    };

    // --- Hook SPA navigation ---
    function processStateChange() {
        const movieId = extractMovieIdFromUrl();
        if (movieId && window.__WordWatch) {
            window.__WordWatch.activateManifest(movieId);
        }
    }

    const hookHistory = (method) => {
        const original = history[method];
        history[method] = function (state, ...args) {
            original.call(history, state, ...args);
            processStateChange();
        };
    };
    hookHistory("pushState");
    hookHistory("replaceState");

    // Helper: extract movieId from current URL
    function extractMovieIdFromUrl() {
        const match = location.pathname.match(/\/watch\/(\d+)/);
        return match ? match[1] : null;
    }
})();