/**
 * Netflix Video API
 * 
 * This module provides a stable API for interacting with Netflix video player.
 * All Netflix-specific DOM queries and interactions are centralized here
 * to make it easier to update when Netflix changes their UI structure.
 */

/**
 * Get the current Netflix video element
 * @returns {HTMLVideoElement|null} The video element or null if not found
 */
function getVideoElement() {
    // Try multiple selectors as Netflix UI evolves
    return document.querySelector('#appMountPoint video') ||
        document.querySelector('.watch-video video') ||
        document.querySelector('video');
}

/**
 * Check if a video is currently playing
 * @returns {boolean} True if video is playing, false otherwise
 */
function isVideoPlaying() {
    const video = getVideoElement();
    if (!video) return false;

    return !video.paused && !video.ended && video.readyState > 2;
}

/**
 * Pause the current video
 * @returns {boolean} True if video was paused, false if no video found
 */
function pauseVideo() {
    const video = getVideoElement();
    if (!video) return false;

    video.pause();
    return true;
}

/**
 * Play the current video
 * @returns {boolean} True if video was played, false if no video found
 */
function playVideo() {
    const video = getVideoElement();
    if (!video) return false;

    video.play();
    return true;
}

/**
 * Get the current video time in seconds
 * @returns {number|null} Current time in seconds or null if no video
 */
function getCurrentTime() {
    const video = getVideoElement();
    return video ? video.currentTime : null;
}

/**
 * Get the video duration in seconds
 * @returns {number|null} Duration in seconds or null if no video
 */
function getVideoDuration() {
    const video = getVideoElement();
    return video ? video.duration : null;
}

/**
 * Check if we're on a Netflix watch page
 * @returns {boolean} True if on a watch page
 */
function isWatchPage() {
    return window.location.pathname.includes('/watch/');
}

/**
 * Get the video ID from the URL
 * @returns {string|null} Video ID or null if not found
 */
function getVideoIdFromUrl() {
    const match = window.location.pathname.match(/\/watch\/(\d+)/);
    return match ? match[1] : null;
}

/**
 * Get video title from Netflix page
 * Tries multiple methods to extract the title
 * @returns {string|null} Video title or null if not found
 */
async function getVideoTitle() {
    // console.debug("[NetflixAPI] Starting title extraction...");

    // 0. Try Preloaded State (Most Reliable & Fast)
    try {
        const response = await new Promise(resolve =>
            chrome.runtime.sendMessage({ action: "GET_NETFLIX_PRELOADED_STATE" }, resolve)
        );
        if (response && response.success && response.title) {
            // console.debug("[NetflixAPI] Preloaded State success:", response.title);
            return response.title;
        }
    } catch (e) {
        // console.warn("[NetflixAPI] Preloaded State extraction failed", e);
    }

    // 1. Try DOM title (Tag agnostic, prioritized)
    const simpleTitle = document.querySelector('[data-uia="video-title"]')?.textContent?.trim();
    if (simpleTitle) {
        // console.debug("[NetflixAPI] DOM title found (data-uia):", simpleTitle);
        return simpleTitle;
    }

    const titleEl = document.querySelector('.video-title h4') ||
        document.querySelector('.ellipsize-text h4');

    if (titleEl?.textContent) {
        // console.debug("[NetflixAPI] DOM title found (legacy):", titleEl.textContent.trim());
        return titleEl.textContent.trim();
    }

    // 2. Fallback: document.title
    // Format: "Movie Name - Netflix"
    if (document.title && document.title.includes(' - Netflix')) {
        // console.debug("[NetflixAPI] Document title found:", document.title);
        return document.title.replace(' - Netflix', '').trim();
    }

    // 3. Keep Deep Extraction as a robust fallback
    try {
        // console.debug("[NetflixAPI] Attempting Deep Extraction...");
        const response = await new Promise(resolve =>
            chrome.runtime.sendMessage({ action: "GET_NETFLIX_TITLE" }, resolve)
        );
        if (response && response.success && response.title) {
            // console.debug("[NetflixAPI] Deep Extraction success:", response.title);
            return response.title;
        }
    } catch (e) {
        console.warn("[NetflixAPI] Deep title extraction failed", e);
    }

    // --- RESTORED METHODS ---

    // Method 4: JSON-LD (Schema.org metadata)
    try {
        const jsonLd = document.querySelector('script[type="application/ld+json"]');
        if (jsonLd) {
            // console.debug("[NetflixAPI] Found JSON-LD script");
            const data = JSON.parse(jsonLd.innerText);
            if (data.name && data.name !== "Netflix") {
                // console.debug("[NetflixAPI] JSON-LD success:", data.name);
                return data.name;
            }
        }
    } catch (e) { console.error("[NetflixAPI] JSON-LD error", e); }

    // Method 5: Meta Tags
    const metaTitle = document.querySelector('meta[property="og:title"]')?.content ||
        document.querySelector('meta[name="twitter:title"]')?.content;

    if (metaTitle && metaTitle !== "Netflix" && !metaTitle.includes("Netflix - Watch")) {
        // console.debug("[NetflixAPI] Meta Tag success:", metaTitle);
        return metaTitle;
    }

    // Method 6: Meta Description Regex
    const desc = document.querySelector('meta[name="description"]')?.content;
    if (desc) {
        const match = desc.match(/^Watch (.*?) \|/);
        if (match && match[1]) {
            // console.debug("[NetflixAPI] Description Regex success:", match[1]);
            return match[1];
        }
    }

    // Method 7: Other DOM Selectors
    const selectors = [
        '.video-title h4',
        // '[data-uia="video-title"]', // Already in #1
        // '.ellipsize-text h4',      // Already in #1
        'div[data-uia="video-title"] span',
        '.PlayerControls--container .video-title',
        'h4.title',
        '.player-status-main-title',
        'span.nf-icon-button-label'
    ];

    for (const sel of selectors) {
        const el = document.querySelector(sel);
        if (el && el.innerText.trim()) {
            // console.debug("[NetflixAPI] Found DOM title with selector:", sel, el.innerText.trim());
            return el.innerText.trim();
        }
    }

    // Method 8: Document Title Fallback (Generic)
    if (document.title && document.title !== "Netflix") {
        return document.title.replace(" - Netflix", "");
    }

    // 9. Fallback: URL parsing with generic name
    const match = location.pathname.match(/\/watch\/(\d+)/);
    if (match) {
        return `Netflix Title (ID: ${match[1]})`;
    }

    return "Netflix Video";
}

/**
 * Observe Netflix title changes
 * @param {Function} callback - Called with new title when changed
 */
function observeNetflixTitle(callback) {
    let lastTitle = "";

    const checkTitle = async () => {
        const title = await getVideoTitle();
        if (title && title !== lastTitle && title !== "Netflix Video") {
            lastTitle = title;
            callback(title);
        }
    };

    // Check immediately
    checkTitle();

    // Observe
    let debounceTimer;
    const observer = new MutationObserver(() => {
        if (debounceTimer) clearTimeout(debounceTimer);
        debounceTimer = setTimeout(checkTitle, 1000);
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true,
        attributes: true,
        characterData: true
    });

    return observer;
}

/**
 * Get current video information
 * @param {Object} wordWatchManager - The WordWatch manager instance (optional)
 * @returns {Object} Object containing videoId and title
 */
async function getCurrentVideoInfo(wordWatchManager) {
    const videoId = wordWatchManager?.lastMovieId || getVideoIdFromUrl();
    const title = (await getVideoTitle()) || 'Unknown Video';

    return {
        videoId: videoId,
        title: title
    };
}

/**
 * Get the Netflix player container element
 * @returns {HTMLElement|null} Player container or null if not found
 */
function getPlayerContainer() {
    return document.querySelector('div[data-uia="video-canvas"]') ||
        document.querySelector('.watch-video') ||
        document.querySelector('.video-container') ||
        document.querySelector('#appMountPoint');
}

/**
 * Check if Netflix controls are active/visible
 * @returns {boolean} True if controls are visible
 */
function areControlsActive() {
    // Try multiple selectors for different Netflix UI versions
    const selectors = [
        '.controls',
        'div[data-uia="controls-standard"]',
        '.watch-video--bottom-controls-container',
        '.PlayerControlsNeo__layout'
    ];

    for (const selector of selectors) {
        const controlsElem = document.querySelector(selector);
        if (controlsElem) {
            // Check if it's the Neo style (newer version)
            if (controlsElem.classList.contains('PlayerControlsNeo__layout')) {
                return !controlsElem.classList.contains('PlayerControlsNeo__layout--inactive');
            }
            return true;
        }
    }

    return false;
}

/**
 * Get video element dimensions
 * @returns {Object|null} Object with width and height or null
 */
function getVideoDimensions() {
    const video = getVideoElement();
    if (!video) return null;

    const rect = video.getBoundingClientRect();
    return {
        width: rect.width,
        height: rect.height
    };
}

/**
 * Check if video is ready to play
 * @returns {boolean} True if video is ready
 */
function isVideoReady() {
    const video = getVideoElement();
    return video && video.readyState >= 2;
}

/**
 * Wait for video element to be available
 * @param {number} timeout - Maximum time to wait in milliseconds
 * @returns {Promise<HTMLVideoElement>} Promise that resolves with video element
 */
function waitForVideoElement(timeout = 10000) {
    return new Promise((resolve, reject) => {
        let elapsed = 0;
        const interval = 200;

        const checkInterval = setInterval(() => {
            const video = getVideoElement();
            if (video) {
                clearInterval(checkInterval);
                resolve(video);
                return;
            }

            elapsed += interval;
            if (elapsed >= timeout) {
                clearInterval(checkInterval);
                reject(new Error('Video element not found within timeout'));
            }
        }, interval);
    });
}

/**
 * Wait for player container to be available
 * @param {number} timeout - Maximum time to wait in milliseconds
 * @returns {Promise<HTMLElement>} Promise that resolves with container element
 */
function waitForPlayerContainer(timeout = 10000) {
    return new Promise((resolve, reject) => {
        let elapsed = 0;
        const interval = 200;

        const checkInterval = setInterval(() => {
            const container = getPlayerContainer();
            if (container) {
                clearInterval(checkInterval);
                resolve(container);
                return;
            }

            elapsed += interval;
            if (elapsed >= timeout) {
                clearInterval(checkInterval);
                reject(new Error('Player container not found within timeout'));
            }
        }, interval);
    });
}

// Export all functions
const NetflixAPI = {
    // Core video functions
    getVideoElement,
    isVideoPlaying,
    pauseVideo,
    playVideo,
    getCurrentTime,
    getVideoDuration,

    // Navigation and page info
    isWatchPage,
    getVideoIdFromUrl,
    getVideoTitle,
    getCurrentVideoInfo,
    observeNetflixTitle,

    // Player UI functions
    getPlayerContainer,
    areControlsActive,

    // Utility functions
    getVideoDimensions,
    isVideoReady,
    waitForVideoElement,
    waitForPlayerContainer
};

// Expose globally for testing and console access
if (typeof window !== 'undefined') {
    window.NetflixAPI = NetflixAPI;
}
