// Add this to the beginning of popup.js

// Check authentication before showing main UI
async function checkAuth() {
    const result = await chrome.storage.local.get(['supabase_session']);

    if (!result.supabase_session) {
        // Not signed in, redirect to sign in page
        window.location.href = 'signin.html';
        return false;
    }

    return true;
}

// Add header buttons (Dashboard & Sign Out)
function addHeaderButtons() {
    const header = document.querySelector('.header');
    const controls = header.querySelector('div[style*="display: flex"]');

    // Create container for new buttons
    const btnContainer = document.createElement('div');
    btnContainer.style.display = 'flex';
    btnContainer.style.alignItems = 'center';
    btnContainer.style.gap = '5px';
    btnContainer.style.marginRight = '10px';
    btnContainer.style.marginLeft = 'auto'; // Push to right if needed, but flex gap handles it

    // --- FAQ BUTTON ---
    const faqBtn = document.createElement('button');
    faqBtn.id = 'openFaq';
    faqBtn.style.cssText = `
        background: none;
        border: none;
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        color: #666;
        transition: color 0.2s;
    `;
    faqBtn.title = 'Help / FAQ';
    faqBtn.innerHTML = `
        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <circle cx="12" cy="12" r="10"></circle>
            <path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"></path>
            <line x1="12" y1="17" x2="12.01" y2="17"></line>
        </svg>
    `;
    faqBtn.addEventListener('click', () => {
        chrome.tabs.create({ url: 'https://www.lingopause.com/faq' });
    });

    // --- DASHBOARD BUTTON ---
    const dashboardBtn = document.createElement('button');
    dashboardBtn.id = 'openDashboard';
    dashboardBtn.style.cssText = `
        background: none;
        border: none;
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        color: #666;
        transition: color 0.2s;
    `;
    dashboardBtn.title = 'Open Dashboard';
    dashboardBtn.innerHTML = `
        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <rect x="3" y="3" width="7" height="7"></rect>
            <rect x="14" y="3" width="7" height="7"></rect>
            <rect x="14" y="14" width="7" height="7"></rect>
            <rect x="3" y="14" width="7" height="7"></rect>
        </svg>
    `;
    dashboardBtn.addEventListener('click', () => {
        const url = (typeof CONFIG !== 'undefined' && CONFIG.DASHBOARD_URL) ? CONFIG.DASHBOARD_URL : 'http://localhost:3000/dashboard';
        chrome.tabs.create({ url: url });
    });

    // --- SIGN OUT BUTTON ---
    const signOutBtn = document.createElement('button');
    signOutBtn.id = 'signOut';
    signOutBtn.style.cssText = `
        background: none;
        border: none;
        cursor: pointer;
        padding: 4px;
        display: flex;
        align-items: center;
        color: #666;
        transition: color 0.2s;
    `;
    signOutBtn.title = 'Sign Out';
    signOutBtn.innerHTML = `
        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path>
            <polyline points="16 17 21 12 16 7"></polyline>
            <line x1="21" y1="12" x2="9" y2="12"></line>
        </svg>
    `;

    signOutBtn.addEventListener('click', async () => {
        if (confirm('Are you sure you want to sign out?')) {
            // Clear session
            await chrome.storage.local.remove(['supabase_session']);

            // Clear Google token if exists
            chrome.identity.getAuthToken({ interactive: false }, (token) => {
                if (token) {
                    chrome.identity.removeCachedAuthToken({ token }, () => {
                        window.location.href = 'signin.html';
                    });
                } else {
                    window.location.href = 'signin.html';
                }
            });
        }
    });

    btnContainer.appendChild(faqBtn);
    btnContainer.appendChild(dashboardBtn);
    btnContainer.appendChild(signOutBtn);

    // Insert before the existing controls (refresh + toggle)
    // The structure is Header -> [Logo/Text] ... [New Buttons] [Refresh/Toggle]
    // Actually current structure is: Header -> [Logo] [Controls Div]
    // We want to insert these buttons BEFORE the Controls Div or INSIDE the Controls Div?
    // User interface requested: "LingoPause Sign Out(icon) Refresh Toggle"
    // Let's modify the header layout slightly.

    // Inserting before the refresh/toggle container
    controls.parentNode.insertBefore(btnContainer, controls);
}

// Modify the DOMContentLoaded event
const originalDOMContentLoaded = document.addEventListener;
document.addEventListener = function (event, callback) {
    if (event === 'DOMContentLoaded') {
        originalDOMContentLoaded.call(this, event, async function (e) {
            const isAuthenticated = await checkAuth();
            if (isAuthenticated) {
                addHeaderButtons();
                callback(e);
            }
        });
    } else {
        originalDOMContentLoaded.call(this, event, callback);
    }
};



document.addEventListener('DOMContentLoaded', () => {
    const activateToggle = document.getElementById('activateToggle');
    const learnLang = document.getElementById('learnLang');
    const knowLang = document.getElementById('knowLang');
    const decreaseFont = document.getElementById('decreaseFont');
    const increaseFont = document.getElementById('increaseFont');
    const fontSizeDisplay = document.getElementById('fontSizeDisplay');
    const bgMode = document.getElementById('bgMode');
    const bgOpacity = document.getElementById('bgOpacity');
    const opacityDisplay = document.getElementById('opacityDisplay');
    // New Toggles
    const showKnowLang = document.getElementById('showKnowLang');
    const showTransliteration = document.getElementById('showTransliteration');

    // New UI Elements
    const langDisplayView = document.getElementById('langDisplayView');
    const langEditView = document.getElementById('langEditView');
    const learnLangDisplay = document.getElementById('learnLangDisplay');
    const knowLangDisplay = document.getElementById('knowLangDisplay');
    const editLangBtn = document.getElementById('editLangBtn');
    const cancelLangBtn = document.getElementById('cancelLangBtn');
    const saveLangBtn = document.getElementById('saveLangBtn');

    let currentFontSize = 24;

    // Helper to get language name from code
    const getLangName = (code) => {
        const lang = LANGUAGES.find(l => l.code === code);
        return lang ? lang.name : 'Not set';
    };

    // Helper to update display text
    const updateLanguageDisplay = () => {
        learnLangDisplay.textContent = getLangName(learnLang.value);
        knowLangDisplay.textContent = getLangName(knowLang.value);
    };

    // Toggle between View and Edit modes
    const toggleLanguageMode = (isEdit) => {
        if (isEdit) {
            langDisplayView.style.display = 'none';
            langEditView.style.display = 'block';
        } else {
            langDisplayView.style.display = 'block';
            langEditView.style.display = 'none';
        }
    };

    // Check for usage limit
    const checkUsageLimit = () => {
        chrome.storage.local.get(['userUsageLimits', 'supabase_session'], (result) => {
            console.debug("[LingoPause] Popup checkUsageLimit:", result);
            const banner = document.getElementById('usageLimitBanner');
            if (banner) {
                const userUsageLimits = result.userUsageLimits || {};
                const currentUserId = result.supabase_session?.user?.id;

                // Show banner only if limit is reached for CURRENT user
                const shouldShow = currentUserId && userUsageLimits[currentUserId];
                console.debug("[LingoPause] Popup shouldShow Limit Banner:", shouldShow, "UserID:", currentUserId);
                banner.style.display = shouldShow ? 'block' : 'none';
            }
        });
    };
    checkUsageLimit();

    // Listen for stored changes
    chrome.storage.onChanged.addListener((changes, namespace) => {
        if (namespace === 'local' && (changes.userUsageLimits || changes.supabase_session)) {
            checkUsageLimit();
        }
    });

    // Populate dropdowns
    const populateSelect = (selectElement) => {
        LANGUAGES.forEach(lang => {
            const option = document.createElement('option');
            option.value = lang.code;
            option.textContent = lang.name;
            selectElement.appendChild(option);
        });
    };

    populateSelect(learnLang);
    populateSelect(knowLang);

    // Load saved settings
    chrome.storage.sync.get(['isActive', 'learnLang', 'knowLang', 'fontSize', 'bgMode', 'bgOpacity', 'showKnowLang', 'showTransliteration'], (result) => {
        if (result.isActive !== undefined) {
            activateToggle.checked = result.isActive;
        }
        if (result.learnLang) {
            learnLang.value = result.learnLang;
        }
        if (result.knowLang) {
            knowLang.value = result.knowLang;
        }
        updateLanguageDisplay(); // Initial update of display text
        if (result.fontSize) {
            currentFontSize = result.fontSize;
            fontSizeDisplay.textContent = `${currentFontSize}px`;
        }
        if (result.bgMode) {
            bgMode.value = result.bgMode;
        }
        if (result.bgOpacity !== undefined) {
            bgOpacity.value = result.bgOpacity;
            opacityDisplay.textContent = `${result.bgOpacity}%`;
        }
        // Load visibility settings (default true if undefined)
        if (result.showKnowLang !== undefined) {
            showKnowLang.checked = result.showKnowLang;
        }
        if (result.showTransliteration !== undefined) {
            showTransliteration.checked = result.showTransliteration;
        }
    });

    // Debounce Utility
    const debounce = (func, wait) => {
        let timeout;
        return (...args) => {
            clearTimeout(timeout);
            timeout = setTimeout(() => func.apply(this, args), wait);
        };
    };

    // Save settings on change
    const saveSettings = () => {
        chrome.storage.sync.set({
            isActive: activateToggle.checked,
            learnLang: learnLang.value,
            knowLang: knowLang.value,
            fontSize: currentFontSize,
            bgMode: bgMode.value,
            bgOpacity: parseInt(bgOpacity.value),
            showKnowLang: showKnowLang.checked,
            showTransliteration: showTransliteration.checked
        }, () => {
            console.debug('[LingoPause] Settings saved');
        });
    };

    decreaseFont.addEventListener('click', () => {
        if (currentFontSize > 12) {
            currentFontSize -= 2;
            fontSizeDisplay.textContent = `${currentFontSize}px`;
            debouncedSaveSettings();
        }
    });

    increaseFont.addEventListener('click', () => {
        if (currentFontSize < 80) {
            currentFontSize += 2;
            fontSizeDisplay.textContent = `${currentFontSize}px`;
            debouncedSaveSettings();
        }
    });

    bgOpacity.addEventListener('input', () => {
        opacityDisplay.textContent = `${bgOpacity.value}%`;
    });

    const debouncedSaveSettings = debounce(saveSettings, 1500);

    bgOpacity.addEventListener('change', debouncedSaveSettings);
    bgMode.addEventListener('change', debouncedSaveSettings);
    // Use immediate save for the main toggle to ensure responsiveness
    activateToggle.addEventListener('change', saveSettings);

    // Immediate save for visibility toggles
    showKnowLang.addEventListener('change', saveSettings);
    showTransliteration.addEventListener('change', saveSettings);

    // REMOVED auto-save on change for languages
    // learnLang.addEventListener('change', debouncedSaveSettings);
    // knowLang.addEventListener('change', debouncedSaveSettings);

    // --- New Event Listeners for Language Flow ---

    editLangBtn.addEventListener('click', () => {
        toggleLanguageMode(true);
    });

    cancelLangBtn.addEventListener('click', () => {
        // Revert dropdowns to saved values (or currently displayed values)
        // Best to just re-read from storage or trust the Selects haven't been "saved" yet,
        // but user might have changed select without saving.
        // Let's reset select values to what's currently in storage/display
        // But simpler: just toggle back, user changes are lost (desired).
        // We should re-sync the selects to the actual 'saved' state to be sure.
        chrome.storage.sync.get(['learnLang', 'knowLang'], (result) => {
            if (result.learnLang) learnLang.value = result.learnLang;
            if (result.knowLang) knowLang.value = result.knowLang;
            toggleLanguageMode(false);
        });
    });

    saveLangBtn.addEventListener('click', () => {
        // 1. Update storage
        saveSettings();
        // 2. Update display
        updateLanguageDisplay();
        // 3. Switch view
        toggleLanguageMode(false);
    });

    const refreshApp = document.getElementById('refreshApp');
    refreshApp.addEventListener('click', () => {
        if (!activateToggle.checked) return;

        // Visual feedback: Rotate the icon
        refreshApp.style.transform = "rotate(180deg)";
        refreshApp.disabled = true;

        // 1. Turn off
        activateToggle.checked = false;
        saveSettings();

        // 2. Turn on after short delay
        setTimeout(() => {
            activateToggle.checked = true;
            saveSettings();

            // Reset rotation and re-enable
            refreshApp.style.transform = "rotate(360deg)";
            setTimeout(() => {
                refreshApp.style.transition = "none";
                refreshApp.style.transform = "rotate(0deg)";
                setTimeout(() => {
                    refreshApp.style.transition = "color 0.2s, transform 0.2s";
                }, 50);
            }, 200);

            refreshApp.disabled = false;
        }, 50);
    });

    // Fetch Updates
    const fetchUpdates = async () => {
        try {
            // Ensure CONFIG is available (requires config.js to be loaded in popup.html)
            const baseUrl = (typeof CONFIG !== 'undefined') ? CONFIG.BACKEND_BASE_URL : 'http://127.0.0.1:8000';

            const updatesContainer = document.getElementById('updatesContainer');
            const updatesContent = document.getElementById('updatesContent');

            const response = await fetch(`${baseUrl}/updates`);
            if (response.ok) {
                const data = await response.json();
                if (data && data.html) {
                    updatesContent.innerHTML = data.html;
                    updatesContainer.style.display = 'block';
                }
            }
        } catch (error) {
            console.error('[LingoPause] Failed to fetch updates:', error);
        }
    };

    fetchUpdates();
});
