import { supabase } from './supabase.js';

document.addEventListener('DOMContentLoaded', async () => {
    const googleSignInBtn = document.getElementById('googleSignIn');
    const emailForm = document.getElementById('emailForm');
    const emailInput = document.getElementById('emailInput');
    const passwordInput = document.getElementById('passwordInput');
    const emailSignInBtn = document.getElementById('emailSignIn');

    // New Elements
    const toggleAuth = document.getElementById('toggleAuth');
    const forgotPasswordLink = document.getElementById('forgotPassword');

    const errorMessage = document.getElementById('errorMessage');
    const successMessage = document.getElementById('successMessage');

    // Restore session
    const { data: { session } } = await supabase.auth.getSession();
    if (session) {
        await chrome.storage.local.set({ supabase_session: session });
        window.location.href = 'popup.html';
        return;
    }

    // =====================
    // Google Sign In
    // =====================
    googleSignInBtn.addEventListener('click', async () => {
        try {
            googleSignInBtn.disabled = true;
            googleSignInBtn.textContent = 'Signing in...';

            const redirectUrl = chrome.identity.getRedirectURL();

            const { data, error } = await supabase.auth.signInWithOAuth({
                provider: 'google',
                options: {
                    redirectTo: redirectUrl,
                    skipBrowserRedirect: true
                }
            });

            if (error) throw error;

            chrome.identity.launchWebAuthFlow(
                {
                    url: data.url,
                    interactive: true
                },
                async (redirectUrl) => {
                    if (chrome.runtime.lastError || !redirectUrl) {
                        console.error(chrome.runtime.lastError);
                        showError('Google sign-in canceled');
                        resetGoogleBtn();
                        return;
                    }

                    const urlObj = new URL(redirectUrl);

                    // Supabase usually puts tokens in the hash (#)
                    // But errors might be in query (?) or hash
                    const hashParams = new URLSearchParams(urlObj.hash.substring(1));
                    const searchParams = new URLSearchParams(urlObj.search.substring(1));

                    // Helper to get param from either
                    const getParam = (key) => hashParams.get(key) || searchParams.get(key);

                    const error = getParam('error');
                    const errorDesc = getParam('error_description');

                    if (error || errorDesc) {
                        console.error('OAuth Error:', error, errorDesc);
                        showError(`Login failed: ${errorDesc || error}`);
                        resetGoogleBtn();
                        return;
                    }

                    const accessToken = getParam('access_token');
                    const refreshToken = getParam('refresh_token');

                    if (!accessToken) {
                        console.error('No access token. Redirect URL:', redirectUrl);
                        showError('No access token found. Check Supabase/Google configuration.');
                        resetGoogleBtn();
                        return;
                    }

                    const { data: sessionData, error: sessionError } =
                        await supabase.auth.setSession({
                            access_token: accessToken,
                            refresh_token: refreshToken
                        });

                    if (sessionError) throw sessionError;

                    await chrome.storage.local.set({
                        supabase_session: sessionData.session
                    });

                    window.location.href = 'popup.html';
                }
            );
        } catch (err) {
            showError(err.message || 'Google sign-in failed');
            resetGoogleBtn();
        }
    });

    let isSignUp = false;

    // =====================
    // Toggle Auth Mode
    // =====================
    function updateAuthMode() {
        // Clear messages
        errorMessage.classList.remove('show');
        successMessage.style.display = 'none';

        if (isSignUp) {
            // Switch to Sign Up UI
            emailSignInBtn.textContent = 'Sign Up';
            toggleAuth.innerHTML = 'Already have an account? <span>Sign in</span>';
            forgotPasswordLink.style.display = 'none'; // Hide forgot password in sign up mode
        } else {
            // Switch to Login UI
            emailSignInBtn.textContent = 'Login';
            toggleAuth.innerHTML = 'New user? <span>Sign up</span>';
            forgotPasswordLink.style.display = 'block';
        }
    }

    toggleAuth.addEventListener('click', (e) => {
        e.preventDefault(); // Check if clicking the span or div
        isSignUp = !isSignUp;
        updateAuthMode();
    });

    // =====================
    // Forgot Password
    // =====================
    forgotPasswordLink.addEventListener('click', async (e) => {
        e.preventDefault();
        const email = emailInput.value.trim();

        if (!email) {
            return showError('Please enter your email to reset password');
        }

        try {
            forgotPasswordLink.style.pointerEvents = 'none';
            forgotPasswordLink.textContent = 'Sending...';

            const { error } = await supabase.auth.resetPasswordForEmail(email, {
                redirectTo: chrome.identity.getRedirectURL() // Optional, but usually needed for deep linking back
            });

            if (error) throw error;

            showSuccess('Password reset link sent to your email.');
        } catch (err) {
            showError(err.message || 'Failed to send reset email');
        } finally {
            forgotPasswordLink.style.pointerEvents = 'auto';
            forgotPasswordLink.textContent = 'Forgot Password?';
        }
    });

    // =====================
    // Email/Password Auth
    // =====================
    emailForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const email = emailInput.value.trim();
        const password = passwordInput.value;

        if (!email) return showError('Please enter your email');
        if (!password) return showError('Please enter your password');

        try {
            emailSignInBtn.disabled = true;
            emailSignInBtn.textContent = isSignUp ? 'Signing up...' : 'Logging in...';
            errorMessage.classList.remove('show');
            successMessage.style.display = 'none';

            if (isSignUp) {
                // --- SIGN UP ---
                const { data, error } = await supabase.auth.signUp({
                    email,
                    password
                });

                if (error) throw error;

                if (data.user && !data.session) {
                    // Confirmation email sent
                    showSuccess('Account created! Please check your email to confirm your account.');
                    emailSignInBtn.textContent = 'Sign Up';
                    // Don't switch mode immediately, let them see message
                } else if (data.session) {
                    // Auto-login
                    await chrome.storage.local.set({ supabase_session: data.session });
                    window.location.href = 'popup.html';
                }
            } else {
                // --- SIGN IN ---
                const { data, error } = await supabase.auth.signInWithPassword({
                    email,
                    password
                });

                if (error) throw error;

                if (data.session) {
                    await chrome.storage.local.set({ supabase_session: data.session });
                    window.location.href = 'popup.html';
                }
            }

        } catch (err) {
            console.error(err);
            showError(err.message || 'Authentication failed');
            emailSignInBtn.textContent = isSignUp ? 'Sign Up' : 'Login';
        } finally {
            emailSignInBtn.disabled = false;
        }
    });

    function resetGoogleBtn() {
        googleSignInBtn.disabled = false;
        googleSignInBtn.textContent = 'Continue with Google';
    }

    function showError(message) {
        errorMessage.textContent = message;
        errorMessage.classList.add('show');
        // Hide success message if showing error
        successMessage.style.display = 'none';

        // Auto-hide error after 5s
        setTimeout(() => errorMessage.classList.remove('show'), 5000);
    }

    function showSuccess(message) {
        successMessage.textContent = message;
        successMessage.style.display = 'block';
        errorMessage.classList.remove('show');
    }
});
